let savedFavorites = JSON.parse(localStorage.getItem('savedFavorites')) || [];

// ===========================
// Veta extension settings
// ===========================

const VETA_DEFAULT_SETTINGS = {
  // Extra tools menu (removed)
  showExtraMenu: false,
  showExtraRatio: false,
  showExtraQuality: false,
  showExtraJson: false,

  // BotAnswer box under article header
  showBotAnswerBox: true,

  // Editor tools tab (removed)
  showEditorToolsTab: false,

  // "Pin artikkel" + favour ites in header
  showFavourites: true,

    // Chat helpers
  showChatbotPreview: true,     
  showChatbotArticles: true,   
  chatbotProfileIcon: '',      

  // Theme: ("default" | "pink" | "ikea")
  themeAccent: 'default'
};



window.__vetaSettings = window.__vetaSettings || { ...VETA_DEFAULT_SETTINGS };
window.__vetaSettingsLoaded = !!window.__vetaSettingsLoaded;

function loadVetaSettings(callback) {
  if (window.__vetaSettingsLoaded) {
    if (callback) callback(window.__vetaSettings);
    return;
  }

  function finish(stored) {
    const safeStored = stored && typeof stored === 'object' ? stored : {};
    window.__vetaSettings = { ...VETA_DEFAULT_SETTINGS, ...safeStored };
    window.__vetaSettingsLoaded = true;
    if (callback) callback(window.__vetaSettings);
  }

  // 1) Try chrome.storage.sync if available
  if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
    try {
      chrome.storage.sync.get('vetaExtensionSettings', (data) => {
        const stored = (data && data.vetaExtensionSettings) || {};
        finish(stored);
      });
      return;
    } catch (e) {
      console.warn('[Veta] Failed to load settings from chrome.storage, falling back to localStorage', e);
      // fall through to localStorage
    }
  }

  // 2) Fallback: use localStorage (works in MAIN world)
  try {
    const raw = localStorage.getItem('vetaExtensionSettings');
    const parsed = raw ? JSON.parse(raw) : {};
    finish(parsed);
  } catch (e) {
    console.warn('[Veta] Failed to load settings from localStorage, using defaults', e);
    finish({});
  }
}


function saveVetaSettings(partial, callback) {
  const base = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const merged = { ...base, ...partial };
  window.__vetaSettings = merged;

  const done = () => {
    if (callback) callback();
  };

  let usedChrome = false;

  // 1) Try chrome.storage.sync if available
  if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.sync) {
    try {
      usedChrome = true;
      chrome.storage.sync.set({ vetaExtensionSettings: merged }, () => {
        done();
      });
    } catch (e) {
      console.warn('[Veta] Failed to save settings to chrome.storage, falling back to localStorage', e);
      usedChrome = false;
    }
  }

  // 2) Fallback: localStorage
  if (!usedChrome) {
    try {
      localStorage.setItem('vetaExtensionSettings', JSON.stringify(merged));
    } catch (e) {
      console.warn('[Veta] Failed to save settings to localStorage', e);
      // still call done(), just no persistence
    }
    done();
  }
}

function applyAvatarEmojiFromSettings() {
  // Ensure settings are loaded first
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { applyAvatarEmojiFromSettings(); } catch (e) {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const emoji = (settings.chatbotProfileIcon || '').trim();

  // If nothing is set, leave the default avatar alone
  if (!emoji) return;

  const avatarImg = document.querySelector(
    '.cwds-user-profile__avatar img.avatar__image'
  );

  if (!avatarImg) {
    // Not on a page with an avatar
    return;
  }

  const avatarRoot = avatarImg.closest('.cwds-user-profile__avatar');
  const aspectBox = avatarImg.closest('.aspect-ratio-box');

  if (avatarRoot) {
    avatarRoot.style.overflow = 'visible';
    avatarRoot.style.borderRadius = '0';
    avatarRoot.style.position = 'relative';
    avatarRoot.style.zIndex = '2';
    avatarRoot.style.backgroundColor = 'transparent';
  }
  if (aspectBox) {
    aspectBox.style.overflow = 'visible';
    aspectBox.style.borderRadius = '0';
    aspectBox.style.backgroundColor = 'transparent';
  }

  const svg = `
    <svg xmlns="http://www.w3.org/2000/svg" width="480" height="480">
      <rect width="100%" height="100%" rx="999" ry="999" fill="transparent"/>
      <text x="50%" y="85%" font-size="500" text-anchor="middle">${emoji}</text>
    </svg>
  `;

  const dataURL = 'data:image/svg+xml,' + encodeURIComponent(svg);

  avatarImg.src = dataURL;
  avatarImg.srcset = dataURL;
  avatarImg.style.position = 'relative';
  avatarImg.style.top = '16px';
  avatarImg.style.zIndex = '3';
  avatarImg.style.borderRadius = '0';
  avatarImg.style.overflow = 'visible';
  avatarImg.style.backgroundColor = 'transparent';
}



/* ===========================
   Veta botAnswer logger (MAIN world)
   =========================== */
(() => {
  if (window.__vetaBotAnswerLoggerInstalled) return;
  window.__vetaBotAnswerLoggerInstalled = true;

const TARGET_PATH = "/document-gql";

const isTargetUrl = (href) => {
  try {
    const u = new URL(href, location.href);

    // Godta alt som slutter på /document-gql (også proxya/relative kall)
    return u.pathname.endsWith(TARGET_PATH);
  } catch {
    return false;
  }
};


  function prettyLog(payload) {
    // forward to UI box if it's been installed
    if (window.__vetaPrettyLog) window.__vetaPrettyLog(payload);

    const doc = payload && payload.data && payload.data.GetDoc;
    console.groupCollapsed("%c[Veta botAnswer]", "color:#0b5; font-weight:600;");

    if (!doc) {
      console.log("(No data.GetDoc in payload)", payload);
      console.groupEnd();
      return;
    }

    if (doc.title) console.log("title:", doc.title);
    if (doc.guid) console.log("guid:", doc.guid);
    if (doc.changeDate) console.log("changeDate:", doc.changeDate);
    if (doc.botAnswer) console.log("botAnswer (plain):", doc.botAnswer);

    if (doc.botAnswerHtml) {
      try {
        const tmp = document.createElement("div");
        tmp.innerHTML = doc.botAnswerHtml;
        const textOnly = (tmp.textContent || "").trim().replace(/\s+/g, " ");
        console.log("botAnswer (from HTML, text):", textOnly);
        console.log("botAnswerHtml (raw):", doc.botAnswerHtml);
      } catch {
        console.log("botAnswerHtml (raw):", doc.botAnswerHtml);
      }
    }

    if (!doc.botAnswer && !doc.botAnswerHtml) {
      console.log("(No botAnswer fields present)", doc);
    }

    console.groupEnd();

    // Also push into the persistent box, if available
    if (window.__vetaUpdateBotAnswerBox && doc) window.__vetaUpdateBotAnswerBox(doc);
  }

  // ---- Patch fetch (page world) ----
  const _fetch = window.fetch;
  function patchedFetch(...args) {
    const ret = _fetch.apply(this, args);
    try {
      const input = args[0];
      const href =
        typeof input === "string"
          ? new URL(input, location.href).href
          : (input && typeof input.url === "string")
            ? new URL(input.url, location.href).href
            : "";

            if (href && href.includes("document-gql")) {
  console.log("[Veta logger] Intercepted fetch to:", href);
}

      if (href && isTargetUrl(href)) {
        ret.then(res => {
          try { res.clone().json().then(prettyLog).catch(() => {}); } catch {}
        }).catch(() => {});
      }
    } catch {}
    return ret;
  }
  window.fetch = patchedFetch;
  try { globalThis.fetch = patchedFetch; } catch {}
  try { self.fetch = patchedFetch; } catch {}

  // ---- Patch XHR as a fallback ----
  const XO = XMLHttpRequest.prototype.open;
  const XS = XMLHttpRequest.prototype.send;

  XMLHttpRequest.prototype.open = function(method, url, ...rest) {
    try { this.__vetaIsTarget = isTargetUrl(String(url)); } catch { this.__vetaIsTarget = false; }
    return XO.call(this, method, url, ...rest);
  };

  XMLHttpRequest.prototype.send = function(...sendArgs) {
    if (this.__vetaIsTarget) {
      this.addEventListener("readystatechange", function() {
        if (this.readyState === 4) {
          try {
            const t = this.responseText;
            if (t && t[0] === "{") prettyLog(JSON.parse(t));
          } catch {}
        }
      });
    }
    return XS.apply(this, sendArgs);
  };

  console.log("%c[Veta logger] installed in PAGE (MAIN) world", "color:#0b5;");
})();

/* ===========================
   Persistent Veta botAnswer box (survives React rerenders)
   =========================== */
(() => {
  if (window.__vetaBotAnswerBoxInit) return;
  window.__vetaBotAnswerBoxInit = true;

  const ANCHOR_SELECTOR = ".buttons-document.flex.items-center";

let _boxHost = null;  
let _shadow = null;   
let _contentEl = null; 
let _titleEl = null;   
let _attachedTo = null;
let _lastGuid = null;
let _lastDocForBox = null;      
let _pendingRenderTimer = 0;

// 🟢 FLYTTET OPP HIT
let _viewMode = "render";
let _isEditing = false;
let _editorEl = null;
let _editedPlainText = false;
let _editBtn = null;

let _lastPayload = { plainText: "", html: "", rawHtml: "" };




// ====== Chat preview støtte ======

function ensureChatPreviewStyles() {
  if (document.getElementById("veta-chatbot-preview-style")) return;

  const style = document.createElement("style");
  style.id = "veta-chatbot-preview-style";

  style.textContent = `

    /* ========================================
       IKEA GLOBAL TOKENS (SPACING + TYPO)
    ========================================= */
    :root {
      --ikea-space-1: 4px;
      --ikea-space-2: 8px;
      --ikea-space-3: 12px;
      --ikea-space-4: 16px;
      --ikea-space-5: 24px;
      --ikea-space-6: 32px;

      --ikea-font-body-s: .875rem; /* 14px */
      --ikea-font-body-m: 1rem;
      --ikea-line-height: 1.571;
    }

    #veta-chatbot-preview,
    #veta-chatbot-preview * {
      font-family: "Noto IKEA","Noto Sans","Roboto","Open Sans",system-ui,sans-serif !important;
      line-height: var(--ikea-line-height);
      box-sizing: border-box;
    }

    /* ========================================
       CHAT WINDOW WRAPPER
    ========================================= */
    #veta-chatbot-preview {
      position: fixed;
      right: 30px;
      bottom: 30px;
      z-index: 999999;
      display: flex;
    }

    #veta-chatbot-preview .chatbot-window {
      min-width: 450px;
      height: 70vh;
      background: #ffffff;
      box-shadow: 0 4px 12px #0003;
      border-radius: 10px;
      display: flex;
      flex-direction: column;
      overflow: hidden;
    }

/* ========================================
   HEADER + BUTTONS (FIXED VERSION)
========================================= */

#veta-chatbot-preview .chatbot-header {
  background-color: rgb(249, 215, 22);
  padding: var(--ikea-space-5) var(--ikea-space-4);
  display: flex;
  align-items: center;
  justify-content: space-between;   /* tekst til venstre, knapper til høyre */
  color: #000;
  font-weight: 600;
}

/* Container som holder minimise + close */
#veta-chatbot-preview .chatbot-header-buttons {
  display: flex;
  align-items: center;
  gap: var(--ikea-space-2);         /* 8px luft mellom ikonene */
  margin-left: auto;                /* tvinger den HELT til høyre */
}

/* Minimise + X icon style */
#veta-chatbot-preview .chatbot-minimise-icon,
#veta-chatbot-preview .chatbot-close-icon {
  cursor: pointer;
  display: flex;
  align-items: center;
  justify-content: center;
  width: 28px;
  height: 28px;
}

/* SVG-størrelse */
#veta-chatbot-preview .chatbot-minimise-icon svg,
#veta-chatbot-preview .chatbot-close-icon svg {
  width: 22px;
  height: 22px;
}

/* Hover-effekt */
#veta-chatbot-preview .chatbot-minimise-icon:hover,
#veta-chatbot-preview .chatbot-close-icon:hover {
  opacity: 0.7;
}

    /* ========================================
       MESSAGE LIST
    ========================================= */
    #veta-chatbot-preview .chatbot-messages-container {
      list-style: none;
      padding: var(--ikea-space-4) var(--ikea-space-2);
      margin: 0 var(--ikea-space-2);
      max-width: 415px;
      overflow-y: auto;
      overflow-x: hidden;
      overscroll-behavior-y: contain;
      background: #fff;
      height: calc(100% - 120px);
    }

    #veta-chatbot-preview .chatbot-message {
      padding: var(--ikea-space-2) var(--ikea-space-4) 0;
      animation: fadeIn .25s ease-out;
    }

    @keyframes fadeIn {
      from { opacity: 0; transform: translateY(4px); }
      to   { opacity: 1; transform: translateY(0); }
    }

    /* ========================================
       SENT MESSAGE (USER)
    ========================================= */
    #veta-chatbot-preview .chatbot-sent-message {
      background-color: #000;
      color: #fff;
      padding: 8px 12px;
      border-radius: 20px 0 20px 20px;
      max-width: 100%;
      word-break: break-word;
    }

    /* ========================================
       BOT MESSAGE (BUBBLE)
    ========================================= */
    #veta-chatbot-preview .received-message-container {
      display: flex;
      align-items: flex-start;
    }

    #veta-chatbot-preview .chatbot-received-message {
      background: #ffffff;
      border: 1px solid #e4e4e4;
      box-shadow: 0 1px 3px #0000001f;
      border-radius: 24px;
      padding: var(--ikea-space-4) var(--ikea-space-4) var(--ikea-space-5);
      margin-top: var(--ikea-space-2);
      max-width: 100%;
      color: #000;
      font-size: var(--ikea-font-body-s);
      word-break: break-word;
    }

    #veta-chatbot-preview .chatbot-received-message p {
      margin: 0;
      line-height: 24px;
    }

    #veta-chatbot-preview .chatbot-received-message a {
      display: inline-block;
      margin-top: var(--ikea-space-1);
      text-decoration: underline;
    }

    /* ========================================
       SPECIAL SPACING RULES (based on Syndeo output)
    ========================================= */

    /* p + <br> + p */
    #veta-chatbot-preview .chatbot-received-message p + br + p {
      margin-top: var(--ikea-space-3) !important; /* 12px */
    }

    /* <a> + <br><br> + <a> */
    #veta-chatbot-preview .chatbot-received-message a + br + br + a {
      margin-top: var(--ikea-space-3) !important; /* 12px */
    }

    /* Lists */
    #veta-chatbot-preview .chatbot-received-message ol {
      margin-left: 1.4em;
    }

    /* ========================================
       FOOTER
    ========================================= */
  #veta-chatbot-preview .chatbot-footer {
  padding: 8px 10px;
  border-top: 1px solid #ddd;
  display: flex;
  align-items: center;
  gap: 8px;
}

#veta-chatbot-preview .chatbot-textarea-container {
  flex: 1 1 auto;
}

#veta-chatbot-preview .chatbot-text-input {
  width: 100%;
  min-height: 24px;
  max-height: 80px;
  resize: none;
  border-radius: 999px;
  border: 1px solid #ccc;
  padding: 6px 16px;
  outline: none;
  font-size: 0.875rem;
}

  `;

  document.head.appendChild(style);

  // Ensure X-button (if chat is already open)
  const header = document.querySelector("#veta-chatbot-preview .chatbot-header");
  if (header && !header.querySelector(".chatbot-close-btn")) {
    const btn = document.createElement("button");
    btn.className = "chatbot-close-btn";
    btn.textContent = "✕";
    btn.onclick = () => document.getElementById("veta-chatbot-preview")?.remove();
    header.appendChild(btn);
  }
}




const CHAT_CUSTOMER_TEMPLATES = [
  'Hi! Can you help me with {TITLE}?',
  'Hi, I have a question about this – can you explain?',
  'Hi! I’m trying to understand how this works.',
  'Hi, can you help me with a question about this topic?'
];

function pickChatCustomerIntro(title) {
  const base = CHAT_CUSTOMER_TEMPLATES[Math.floor(Math.random() * CHAT_CUSTOMER_TEMPLATES.length)];
  if (!title) return base.replace('{TITLE}', 'dette');
  return base.replace('{TITLE}', title);
}

function openVetaChatPreview() {
  // No botAnswer => nothing to show
  if (
    !_lastPayload ||
    (!(_lastPayload.html || _lastPayload.rawHtml || _lastPayload.plainText))
  ) {
    if (window.showCopyToast) {
      window.showCopyToast('No BotAnswer to show as chatbot');
    }
    return;
  }

  ensureChatPreviewStyles();

  let container = document.getElementById('veta-chatbot-preview');

  if (!container) {
    container = document.createElement('div');
    container.id = 'veta-chatbot-preview';
    container.innerHTML = `
      <div id="chatbot-window" class="chatbot-window chat-right-alignment">
        <div class="chatbot-header">
          <div class="chatbot-header-icon">
            <svg width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
              <path fill-rule="evenodd" clip-rule="evenodd"
                d="M20 3H4H2V5V15V17H4H7V22L14 17L20 17H22V15V5V3H20ZM7 15L9 15V17V18L13 15H14L20 15V5H4L4 15H7Z"
                fill="#000000"></path>
            </svg>
          </div>
          <h1 class="chatbot-header-text" tabindex="0">IKEA Chat (preview)</h1>
          <div class="chatbot-header-buttons">
            <div tabindex="0" aria-label="Minimize" role="button" class="chatbot-minimise-icon">
              <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                <path fill-rule="evenodd" clip-rule="evenodd"
                  d="M10.027 12L15 16.945 19.973 12l1.527 1.522L15 20l-6.5-6.478L10.027 12z"
                  fill="#000000"></path>
              </svg>
            </div>
            <div tabindex="0" aria-label="Close" role="button" class="chatbot-close-icon">
              <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                <path fill-rule="evenodd" clip-rule="evenodd"
                  d="M15 13.6L20.6 8 22 9.4 16.4 15l5.6 5.6-1.4 1.4-5.6-5.6L9.4 22 8 20.6l5.6-5.6L8 9.4 9.4 8l5.6 5.6z"
                  fill="#000000"></path>
              </svg>
            </div>
          </div>
        </div>

        <ul class="chatbot-messages-container anchor-disabled" tabindex="-1" role="log"></ul>

        <div class="chatbot-footer">
          <div class="chatbot-textarea-container">
            <textarea tabindex="0" id="chatbot-text-input" class="chatbot-text-input" placeholder="Type here..."></textarea>
          </div>
          <div class="buttons-container">
            <div tabindex="0" class="chatbot-send-icon send-icon-disabled" role="button" aria-label="Send message"></div>
          </div>
        </div>
        <div class="scroll-indicator" role="presentation" tabindex="-1"></div>
      </div>
    `;
    document.body.appendChild(container);

    const closeBtn = container.querySelector('.chatbot-close-icon');
    const minBtn = container.querySelector('.chatbot-minimise-icon');

    if (closeBtn) {
      closeBtn.addEventListener('click', () => {
        container.remove();
        const s = document.getElementById('veta-chatbot-preview-style');
        if (s) s.remove();
      });
    }

    if (minBtn) {
      minBtn.addEventListener('click', () => {
        container.classList.toggle('veta-chatbot-preview-minimized');
      });
    }
  } else {
    container.style.display = 'flex';
  }

  const list = container.querySelector('.chatbot-messages-container');
  if (!list) return;
  list.innerHTML = '';

  function addSystem(text) {
    const li = document.createElement('li');
    li.className = 'chatbot-message';
    li.innerHTML = `
      <div class="participant-event">
        <span class="participant-message" tabindex="0">${text}</span>
      </div>
    `;
    list.appendChild(li);
  }

  function addCustomer(text) {
    const li = document.createElement('li');
    li.className = 'chatbot-message';
    li.innerHTML = `
      <div class="sent-message-container">
        <div tabindex="0" class="chatbot-sent-message">
          <span class="visually-hidden"></span>
          <span>${text}</span>
        </div>
      </div>
    `;
    list.appendChild(li);
  }

  function addBot(html) {
    const li = document.createElement('li');
    li.className = 'chatbot-message';
    const wrapper = document.createElement('div');
    wrapper.className = 'received-message-container Bot';
    const filler = document.createElement('div');
    const inner = document.createElement('div');
    inner.className = 'chatbot-received-message';
    inner.tabIndex = 0;
    inner.innerHTML = html;
    wrapper.appendChild(filler);
    wrapper.appendChild(inner);
    li.appendChild(wrapper);
    list.appendChild(li);
  }

  // Scriptet som “faker” samtalen

  addSystem('The chatbot is online and ready');

  const title = _lastDocForBox && _lastDocForBox.title ? _lastDocForBox.title : '';
  const introText = pickChatCustomerIntro(title);
  addCustomer(introText);

  const botHtml =
    (_editedPlainText && _lastPayload.plainText)
      ? `<p>${_lastPayload.plainText}</p>`
      : (_lastPayload.html ||
         _lastPayload.rawHtml ||
         (_lastPayload.plainText ? `<p>${_lastPayload.plainText}</p>` : '<p>(no BotAnswer)</p>'));

  addBot(botHtml);

  const input = container.querySelector('#chatbot-text-input');
  if (input) input.focus();
}





  const throttle = (fn, ms) => {
    let timer = 0, savedArgs = null;
    const run = () => {
      timer = 0;
      const a = savedArgs; savedArgs = null;
      fn.apply(null, a);
    };
    return (...args) => {
      savedArgs = args;
      if (!timer) timer = setTimeout(run, ms);
    };
  };

  function getPlainTextFromHtml(html) {
    if (!html) return "";
    try {
      const tmp = document.createElement("div");
      tmp.innerHTML = html;
      return (tmp.textContent || "").trim().replace(/\s+/g, " ");
    } catch {
      return "";
    }
  }

  function applyEditorValueToPayload() {
    if (!_editorEl) return;
    const v = _editorEl.value || "";

    if (_viewMode === "html") {
      _editedPlainText = false;
      _lastPayload.rawHtml = v;
      _lastPayload.html = v || "";
      _lastPayload.plainText = getPlainTextFromHtml(v);
    } else {
      _editedPlainText = true;
      _lastPayload.plainText = v;
    }
  }

  function renderEditorFromPayload() {
    if (!_contentEl) return;

    let seed = "";
    if (_viewMode === "html") {
      seed = _lastPayload.rawHtml || _lastPayload.html || "";
    } else {
      seed =
        _lastPayload.plainText ||
        getPlainTextFromHtml(_lastPayload.html) ||
        getPlainTextFromHtml(_lastPayload.rawHtml);
    }

    _contentEl.innerHTML = "";
    _contentEl.classList.remove("muted");
    _contentEl.classList.toggle("code", _viewMode === "html");

    const ta = document.createElement("textarea");
    ta.className = "editor" + (_viewMode === "html" ? " editor-code" : "");
    ta.value = seed || "";
    ta.addEventListener("input", () => {
      applyEditorValueToPayload();
    });

    _editorEl = ta;
    _contentEl.appendChild(ta);
  }

  function updateEditUI() {
    if (!_editBtn) return;
    _editBtn.textContent = _isEditing ? "Done" : "Edit";
    _editBtn.title = _isEditing ? "Exit edit mode" : "Edit botAnswer";
    _editBtn.classList.toggle("btn-toggle-active", _isEditing);
  }

  function createHostIfNeeded() {
    if (_boxHost && _boxHost.isConnected) return;

    _boxHost = document.createElement("div");
    _boxHost.id = "veta-botanswer-host";
    _shadow = _boxHost.attachShadow({ mode: "open" });

    const style = document.createElement("style");
   style.textContent = `
  :host { all: initial; }
  .box {
    margin-top: 12px;
    padding: 12px 14px;
    background: #f8f9fa;
    border: 1px solid #ddd;
    border-radius: 10px;
    font: 14px/1.45 system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
    color: #222;
    white-space: pre-wrap;
    word-wrap: break-word;
  }
  .header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 8px;
    margin-bottom: 6px;
  }
  .title {
    font-weight: 600;
    opacity: 0.8;
  }
  .actions {
    display: flex;
    gap: 4px;
  }
  .btn {
    border: 1px solid #ccc;
    background: #f1f3f5;
    border-radius: 6px;
    padding: 2px 8px;
    font-size: 11px;
    cursor: pointer;
  }
  .btn:hover {
    background: #e9ecef;
  }
  .btn-toggle-active {
    background: #dee2e6;
    border-color: #999;
  }
  .btn-chat {
    font-weight: 500;
  }
  .content a { text-decoration: underline; }
  .muted { opacity: 0.7; }
  .code {
    font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
    white-space: pre;
    overflow-x: auto;
  }
  .editor {
    width: 100%;
    min-height: 120px;
    resize: vertical;
    border: 1px solid #ddd;
    border-radius: 6px;
    padding: 8px 10px;
    font: 13px/1.45 system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
    background: #fff;
    color: #222;
  }
  .editor-code {
    font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
    white-space: pre;
  }
`;



const wrapper = document.createElement("div");
wrapper.className = "box";
wrapper.innerHTML = `
  <div class="header">
    <div class="title">BotAnswer</div>
    <div class="actions">
      <button type="button" class="btn btn-toggle" title="Show raw HTML">HTML</button>
      <button type="button" class="btn btn-chat" title="Show as chatbot">Show as chatbot</button>
      <button type="button" class="btn btn-edit" title="Edit botAnswer">Edit</button>
      <button type="button" class="btn btn-copy" title="Copy">Copy</button>
    </div>
  </div>
  <div class="content muted">Open an article to show botAnswer…</div>
`;


_titleEl = wrapper.querySelector(".title");
_contentEl = wrapper.querySelector(".content");

const toggleBtn = wrapper.querySelector(".btn-toggle");
const copyBtn = wrapper.querySelector(".btn-copy");
const chatBtn = wrapper.querySelector(".btn-chat");
_editBtn = wrapper.querySelector(".btn-edit");

function updateToggleUI() {
  if (!toggleBtn) return;
  if (_viewMode === "html") {
    toggleBtn.classList.add("btn-toggle-active");
    toggleBtn.textContent = "Show text";
    toggleBtn.title = "Show formatted botAnswer";
  } else {
    toggleBtn.classList.remove("btn-toggle-active");
    toggleBtn.textContent = "Show HTML";
    toggleBtn.title = "Show raw HTML for botAnswer";
  }
}

if (_editBtn) {
  _editBtn.addEventListener("click", () => {
    if (_isEditing) {
      applyEditorValueToPayload();
    }
    _isEditing = !_isEditing;
    updateEditUI();
    if (_lastPayload) setBotAnswer(_lastPayload);
  });
  updateEditUI();
}

if (toggleBtn) {
  toggleBtn.addEventListener("click", () => {
    if (_isEditing) applyEditorValueToPayload();
    _viewMode = _viewMode === "render" ? "html" : "render";
    updateToggleUI();

    // re-render siste payload
    if (_lastPayload) {
      setBotAnswer(_lastPayload);
    }
  });

  updateToggleUI();
}

if (copyBtn) {
  copyBtn.addEventListener("click", () => {
    if (_isEditing) applyEditorValueToPayload();
    let textToCopy = "";

    if (_viewMode === "html" && _lastPayload.rawHtml) {
      // I HTML-modus: kopier rå HTML
      textToCopy = _lastPayload.rawHtml;
    } else if (_lastPayload.plainText) {
      // Ellers: kopier ren tekst hvis vi har det
      textToCopy = _lastPayload.plainText;
    } else if (_lastPayload.html) {
      // Fallback: kopier HTML-feltet
      textToCopy = _lastPayload.html;
    }

    if (!textToCopy) return;

    if (window.copyTextToClipboard) {
      window.copyTextToClipboard(textToCopy)
        .then(() => {
          if (window.showCopyToast) {
            window.showCopyToast('Copied to clipboard ✅');
          }
        })
        .catch(() => {});
    } else {
      // enkel fallback
      const ta = document.createElement("textarea");
      ta.value = textToCopy;
      ta.style.position = "fixed";
      ta.style.top = "-9999px";
      document.body.appendChild(ta);
      ta.select();
      document.execCommand("copy");
      document.body.removeChild(ta);
    }
  });
}
if (chatBtn) {
  chatBtn.addEventListener('click', () => {
    if (_isEditing) applyEditorValueToPayload();
    openVetaChatPreview();
  });
}

_shadow.appendChild(style);
_shadow.appendChild(wrapper);


try { applyChatbotPreviewVisibilityFromSettings(); } catch {}

  }

  function placeBox() {
    const anchor = document.querySelector(ANCHOR_SELECTOR);
    if (!anchor) return;

    createHostIfNeeded();

    // Already placed after this anchor?
    if (_attachedTo === anchor && _boxHost.previousSibling === anchor) return;

    anchor.insertAdjacentElement("afterend", _boxHost);
    _attachedTo = anchor;

    if (_lastDocForBox) {
      renderDocIntoBox(_lastDocForBox);
    }
  }

  const ensurePlaced = throttle(placeBox, 100);

  // SAFE observer startup (avoids "parameter 1 is not of type 'Node'" crash)
  const mo = new MutationObserver(() => ensurePlaced());
  function safeObserve() {
    try {
      const root = document.documentElement || document.body || null;
      if (root && root.nodeType === 1) {
        mo.observe(root, { childList: true, subtree: true });
        return;
      }
      setTimeout(safeObserve, 50);
    } catch {
      setTimeout(safeObserve, 100);
    }
  }
  safeObserve();

  // Initial placement retries
  if ("requestIdleCallback" in window) requestIdleCallback(() => ensurePlaced());
  if ("requestAnimationFrame" in window) requestAnimationFrame(ensurePlaced);
  setTimeout(ensurePlaced, 250);
  setTimeout(ensurePlaced, 750);

  // --- helpers to update box ---

  function setKeywords(keywords) {
    if (!_titleEl) return;
    const list = Array.isArray(keywords) ? keywords.filter(Boolean) : [];
    const short = list.slice(0, 8); // keep it tidy
    const suffix = short.length ? ` — keywords: ${short.join(", ")}` : "";
    _titleEl.textContent = `BotAnswer${suffix}`;
  }

  function setBotAnswer({ plainText, html, rawHtml }) {
    if (!_contentEl || !_boxHost || !_boxHost.isConnected) {
      ensurePlaced();
      if (!_contentEl) return;
    }

    // Husk siste payload for toggle + kopi
    _lastPayload = {
      plainText: plainText || "",
      html: html || "",
      rawHtml: rawHtml || ""
    };

    const hasContent = !!(_lastPayload.plainText || _lastPayload.html || _lastPayload.rawHtml);
    if (!hasContent && !_isEditing) {
      _contentEl.textContent = "(No BotAnswer for this article)";
      _contentEl.classList.add("muted");
      _contentEl.classList.remove("code");
      return;
    }

    _contentEl.classList.remove("muted");

    if (_isEditing) {
      renderEditorFromPayload();
      return;
    }

    // HTML-modus: vis rå HTML som tekst
    if (_viewMode === "html" && _lastPayload.rawHtml) {
      _contentEl.textContent = _lastPayload.rawHtml;
      _contentEl.classList.add("code");
      return;
    }

    // Normal visning: prøv ALLTID å rendre HTML hvis vi har det
    _contentEl.classList.remove("code");

    if (_editedPlainText && _lastPayload.plainText) {
      _contentEl.textContent = _lastPayload.plainText;
      return;
    }

    if (_lastPayload.html) {
      _contentEl.innerHTML = _lastPayload.html;
      return;
    }

    if (_lastPayload.plainText) {
      _contentEl.textContent = _lastPayload.plainText;
      return;
    }

    if (_lastPayload.rawHtml) {
      _contentEl.textContent = _lastPayload.rawHtml;
      return;
    }

    _contentEl.textContent = "(No BotAnswer for this article)";
    _contentEl.classList.add("muted");
  }

  function renderDocIntoBox(d) {
    if (!_contentEl || !_boxHost || !_boxHost.isConnected) {
      return false;
    }
    if (!d) return false;

    let textOnly = "";
    let rawHtml = d.botAnswerHtml || "";

    if (rawHtml) {
      const tmp = document.createElement("div");
      tmp.innerHTML = rawHtml;
      textOnly = (tmp.textContent || "").trim().replace(/\s+/g, " ");
    }

    const plainText = textOnly || (d.botAnswer || "");

    setBotAnswer({
      plainText,
      html: rawHtml || null,   // always use HTML field for rendering
      rawHtml
    });

    // update title/keywords
    setKeywords(d.keywords);

    // save meta for last opened VETA document
    if (d.guid) {
      window.__vetaLastDocMeta = {
        guid: d.guid,
        vetaDocId: String(d.vetaDocId || d.id || ''),
        title: d.title || '',
        marketCode: d.marketCode || 'NO',
      };
    }

    return true;
  }



  // exposed hook used by the logger
  // exposed hook used by the logger
  window.__vetaUpdateBotAnswerBox = (doc) => {
    try {
      if (!doc) return;

      // Husk siste doc
      _lastDocForBox = doc;
      _isEditing = false;
      _editedPlainText = false;
      updateEditUI();

      // Sørg for at boksen blir forsøkt plassert
      ensurePlaced();

      function doRender(d) {
        return renderDocIntoBox(d);
      }

      // 1) Prøv å rendre med en gang
      if (doRender(doc)) return;

      // 2) Hvis boksen ikke er klar ennå, prøv én gang til etter en liten delay
      if (_pendingRenderTimer) clearTimeout(_pendingRenderTimer);
      _pendingRenderTimer = setTimeout(() => {
        _pendingRenderTimer = 0;
        if (_lastDocForBox) {
          doRender(_lastDocForBox);
        }
      }, 200);
    } catch {
      // no-op
    }
  };



  // also let prettyLog call into this with the full payload, if needed later
  window.__vetaPrettyLog = (payload) => {
    const doc = payload && payload.data && payload.data.GetDoc;
    if (!doc) return;
    window.__vetaUpdateBotAnswerBox(doc);
  };
})();

/* ===========================
   Extra UI helpers (your stuff)
   =========================== */

console.log("new update with editor-fix and doc-toggle");

function ensureEditorHeaderStyle() {
  // Bare på /editor
  if (!location.pathname.startsWith('/editor')) return;

  if (document.getElementById('veta-editor-header-style')) return;

  const style = document.createElement('style');
  style.id = 'veta-editor-header-style';
  style.textContent = `
.cwds-header__header.nav-editor {
    background-color: #1652a7ff !important;
  }

    /* Innholdskontaineren inni headeren */
    .cwds-header__container.cwds-header__narrow {
      background-color: transparent !important; /* eller #000 hvis du vil matche helt */
    }
  `;
  document.head.appendChild(style);
}





// --- 2.1: Hide/show right document panel (from bookmarklet) ---

function setDocRightPanelHidden(hidden) {
  const s = document.querySelector('section.mt-5.grid.h-full.w-full');
  if (!s) return;

  let left = s.children[0];
  let right = s.children[1];

  if (!right) right = s.querySelector('div.flex.h-full.flex-col.bg-white');
  if (!left) left = s.querySelector('div.h-full.overflow-y-auto.p-3');
  if (!left || !right) return;

  if (!window.__docOrigGridTemplate) {
    const cs = window.getComputedStyle(s);
    window.__docOrigGridTemplate = cs.gridTemplateColumns;
  }

  if (hidden) {
    s.style.gridTemplateColumns = '1fr';
    right.style.position = 'absolute';
    right.style.left = '-9999px';
    right.style.top = '0';
    right.style.width = '0';
    right.style.height = '0';
    right.style.overflow = 'hidden';
    right.style.pointerEvents = 'none';
    right.style.opacity = '0';
    left.style.gridColumn = '1 / -1';
    left.style.width = '100%';
    left.style.maxWidth = '100%';
  } else {
    s.style.gridTemplateColumns = window.__docOrigGridTemplate || '';
    right.style.position = '';
    right.style.left = '';
    right.style.top = '';
    right.style.width = '';
    right.style.height = '';
    right.style.overflow = '';
    right.style.pointerEvents = '';
    right.style.opacity = '';
    left.style.gridColumn = '';
    left.style.width = '';
    left.style.maxWidth = '';
  }
}

function ensureDocRightToggleButton() {
  const container = document.querySelector('.mb-5.flex.w-full.justify-between .flex.w-full.justify-end');
  if (!container) return;

  let btn = document.getElementById('doc-toggle-right');
  if (!btn) {
    btn = document.createElement('button');
    btn.id = 'doc-toggle-right';
    btn.textContent = 'Hide document info';
    btn.style.padding = '4px 8px';
    btn.style.fontSize = '11px';
    btn.style.border = '1px solid #ccc';
    btn.style.borderRadius = '4px';
    btn.style.background = '#f5f5f5';
    btn.style.cursor = 'pointer';
    btn.style.marginTop = '6px';

    btn.addEventListener('click', () => {
      const current = !!window.__docRightHidden;
      const next = !current;
      window.__docRightHidden = next;
      setDocRightPanelHidden(next);
      btn.textContent = next ? 'Show document info' : 'Hide document info';
    });

    container.appendChild(btn);
  }

  // Sørg for at panelet matcher gjeldende state
  setDocRightPanelHidden(!!window.__docRightHidden);
  btn.textContent = window.__docRightHidden ? 'Show document info' : 'Hide document info';
}









// --- 2.2: Fjern “Editor < create < new”-teksten i header ---

function cleanupEditorBreadcrumbText() {
  const textEls = document.querySelectorAll('a[href="/editor"] .btn__inner .text.text--body-m');
  textEls.forEach(el => el.remove());
}

function removeLiveEditWarning() {
  const candidates = document.querySelectorAll('.text.text--heading-s.p-5.text-red-700');
  candidates.forEach(el => {
    const txt = (el.textContent || '').trim();
    if (txt.startsWith('Live edit:')) {
      el.remove();
    }
  });
}


// --- 2.4: Verktøy-tab i editorens høyre panel (Notater / Åpne URL) ---

// --- 2.4: Verktøy-tab i editorens høyre panel (Notater / Åpne URL) ---

function addEditorToolsTab() {
  // Editor tools are removed.
  return;
  // Run only in the editor
  if (!location.pathname.startsWith('/editor')) return;

  // ---- Helpers ----



  function getOrCreateToolsPanel(tabsRoot) {
    let panel = tabsRoot.querySelector('#tab_7');
    if (!panel) {
      panel = document.createElement('div');
      panel.id = 'tab_7';
      panel.className = 'tabs__panel';
      panel.setAttribute('role', 'tabpanel');
      panel.tabIndex = -1;
      panel.hidden = true;
      tabsRoot.appendChild(panel);
    }
    return panel;
  }

  function setActiveTab(tabsRoot, tabButton, panelId) {
    const tabButtons = tabsRoot.querySelectorAll('.tabs__tab');
    tabButtons.forEach(btn => {
      btn.classList.remove('tabs__tab--active');
      btn.setAttribute('aria-selected', 'false');
    });
    tabButton.classList.add('tabs__tab--active');
    tabButton.setAttribute('aria-selected', 'true');

    const panels = tabsRoot.querySelectorAll('.tabs__panel');
    panels.forEach(p => { p.hidden = true; });

    const targetPanel = tabsRoot.querySelector('#' + panelId);
    if (targetPanel) {
      targetPanel.hidden = false;
    }
  }

  function renderToolsHome(panel) {
    panel.innerHTML = `
      <div class="flex flex-col gap-4 px-4 py-4">
        <div class="text text--heading-s">Tools for this article</div>
        <div style="display:flex; gap:8px; flex-wrap:wrap;">
          <button id="veta-tools-notes-btn"
            style="padding:8px 12px; border:none; border-radius:4px; background:#0058a3; color:#fff; cursor:pointer; font-size:14px;">
            Notes
          </button>
          <button id="veta-tools-url-btn"
            style="padding:8px 12px; border:none; border-radius:4px; background:#666; color:#fff; cursor:pointer; font-size:14px;">
            Open URL
          </button>
        </div>
       
      </div>
    `;

    const notesBtn = panel.querySelector('#veta-tools-notes-btn');
    const urlBtn = panel.querySelector('#veta-tools-url-btn');

    notesBtn.addEventListener('click', () => renderNotesView(panel));
    urlBtn.addEventListener('click', () => renderUrlView(panel));
  }

  // Notater: bare midlertidig, ingen lagring
  function renderNotesView(panel) {
    panel.innerHTML = `
      <div class="flex flex-col gap-4 px-4 py-4">
        <div style="display:flex; justify-content:space-between; align-items:center; gap:8px;">
          <div class="text text--heading-s">Notes for this article</div>
          <button id="veta-notes-back-btn"
            style="padding:4px 8px; border:none; border-radius:4px; background:#eee; cursor:pointer; font-size:12px;">
            ← Back to tools
          </button>
        </div>
        <textarea id="veta-notes-textarea"
          style="width: 100%; min-height: 260px; padding: 8px; border-radius: 4px; border: 1px solid #ccc; resize: vertical; font-size: 14px;"></textarea>
        <div class="text text--body-s" style="color:#666;">
          Notes are stored only while this page is open. Close/refresh to clear them.
        </div>
      </div>
    `;

    const backBtn = panel.querySelector('#veta-notes-back-btn');
    backBtn.addEventListener('click', () => renderToolsHome(panel));
  }

function addIframeExpandControls(panel, wrapper, iframe) {
  if (!panel || !wrapper || !iframe) return;

  // Ikke lag knappen flere ganger
  if (panel.querySelector('#veta-iframe-expand-btn')) return;

  // Knappen under iframe
  const expandBtn = document.createElement('button');
  expandBtn.id = 'veta-iframe-expand-btn';
  expandBtn.textContent = 'Expand';
  Object.assign(expandBtn.style, {
    marginTop: '8px',
    padding: '6px 10px',
    border: 'none',
    borderRadius: '4px',
    background: '#0058a3',
    color: '#fff',
    cursor: 'pointer',
    fontSize: '13px'
  });

  wrapper.insertAdjacentElement('afterend', expandBtn);

  function createOverlay() {
    let overlay = document.getElementById('veta-iframe-overlay');
    if (overlay) return overlay;

    // Hele høyresiden med tabs
    const tabsShell = panel.closest('.tabs.w-full');
    if (!tabsShell) return null;

    // Sørg for at vi kan posisjonere absolutt inni denne
    if (getComputedStyle(tabsShell).position === 'static') {
      tabsShell.style.position = 'relative';
    }

    // Hjelper høyden til å fylle kolonnen
    tabsShell.style.height = '100%';

    overlay = document.createElement('div');
    overlay.id = 'veta-iframe-overlay';
    Object.assign(overlay.style, {
      position: 'absolute',
      inset: '0',              // fyll hele tabsShell
      background: '#fff',
      zIndex: 100,
      display: 'none',
      flexDirection: 'column',
      borderLeft: '1px solid #ddd'
    });

    const header = document.createElement('div');
    Object.assign(header.style, {
      display: 'flex',
      alignItems: 'center',
      justifyContent: 'space-between',
      padding: '4px 6px',
      borderBottom: '1px solid #ddd',
      fontSize: '12px',
      background: '#f5f5f5',
      flex: '0 0 auto'
    });
    header.textContent = 'Large URL view';

    const closeBtn = document.createElement('button');
    closeBtn.textContent = '✕';
    Object.assign(closeBtn.style, {
      padding: '2px 6px',
      border: 'none',
      borderRadius: '4px',
      background: '#eee',
      cursor: 'pointer',
      fontSize: '11px',
      // trekk den litt inn i hjørnet
      marginRight: '-5px',
      marginTop: '-5px'
    });
    closeBtn.addEventListener('click', shrink);

    header.appendChild(closeBtn);

    const body = document.createElement('div');
    body.className = 'veta-iframe-overlay-inner';
    Object.assign(body.style, {
      flex: '1 1 auto',    // la iframen få all tilgjengelig høyde
      display: 'flex',
      minHeight: '0'
    });

    overlay.appendChild(header);
    overlay.appendChild(body);
    tabsShell.appendChild(overlay);

    return overlay;
  }

  function expand() {
    const overlay = createOverlay();
    if (!overlay) return;
    const body = overlay.querySelector('.veta-iframe-overlay-inner');
    if (!body) return;

    // flytt iframe inn i overlay
    body.appendChild(iframe);

    // skjul original boks
    wrapper.style.display = 'none';

    overlay.style.display = 'flex';
    expandBtn.textContent = 'Normal size';
  }

  function shrink() {
    const overlay = document.getElementById('veta-iframe-overlay');
    if (!overlay) return;

    // flytt iframe tilbake
    wrapper.appendChild(iframe);
    wrapper.style.display = '';

    overlay.style.display = 'none';
    expandBtn.textContent = 'Expand';
  }

  let expanded = false;
  expandBtn.addEventListener('click', () => {
    if (!expanded) {
      expand();
    } else {
      shrink();
    }
    expanded = !expanded;
  });
}



  // URL + iframe
function renderUrlView(panel) {
  panel.innerHTML = `
    <div class="flex flex-col gap-3 px-4 py-4" style="height:100%;">
      <div style="display:flex; justify-content:space-between; align-items:center; gap:8px; margin-bottom:4px;">
        <div class="text text--heading-s">Open URL in the side panel</div>
        <button id="veta-url-back-btn"
          style="padding:4px 8px; border:none; border-radius:4px; background:#eee; cursor:pointer; font-size:12px;">
          ← Back to tools
        </button>
      </div>
      <div style="display:flex; gap:8px; align-items:center; margin-bottom:8px;">
        <input id="veta-url-input" type="text" placeholder="https://..."
          style="flex:1; padding:6px 8px; border-radius:4px; border:1px solid #ccc; font-size:14px;">
        <button id="veta-url-open-btn"
          style="padding:6px 10px; border:none; border-radius:4px; background:#0058a3; color:#fff; cursor:pointer; font-size:14px;">
          Open
        </button>
      </div>
    
      <div id="veta-url-iframe-wrapper"
           style="border:1px solid #ddd; border-radius:4px; overflow:hidden; height:57vh;">
        <iframe id="veta-url-iframe" src="" style="width:100%; height:100%; border:none;"></iframe>
      </div>
    </div>
  `;

  const input = panel.querySelector('#veta-url-input');
  const btn = panel.querySelector('#veta-url-open-btn');
  const wrapper = panel.querySelector('#veta-url-iframe-wrapper'); // 👈 denne manglet
  const iframe = panel.querySelector('#veta-url-iframe');
  const backBtn = panel.querySelector('#veta-url-back-btn');

  // Legg til utvid-/krymp-knappen
  addIframeExpandControls(panel, wrapper, iframe);

  function openUrl() {
    let url = (input.value || '').trim();
    if (!url) return;
    if (!/^https?:\/\//i.test(url)) {
      url = 'https://' + url;
    }
    iframe.src = url;
  }

  btn.addEventListener('click', openUrl);
  input.addEventListener('keydown', e => {
    if (e.key === 'Enter') {
      e.preventDefault();
      openUrl();
    }
  });

  backBtn.addEventListener('click', () => renderToolsHome(panel));
}


  // ---- Hoveddel ----
  const tabsRoot = getSideTabsRoot();
  if (!tabsRoot) return;

  // Stram inn tab-raden (på alle tabs, inkl. Metadata / Feedback / osv.)
  shrinkEditorTabBar(tabsRoot);

  const overflowContent = tabsRoot.querySelector('.overflow-carousel__content');
  if (!overflowContent) return;

  // Ikke lag flere enn én Verktøy-tab
  if (overflowContent.querySelector('.veta-tools-tab')) return;

  const toolsTab = document.createElement('button');
  toolsTab.type = 'button';
  toolsTab.className = 'tabs__tab veta-tools-tab';
  toolsTab.setAttribute('role', 'tab');
  toolsTab.setAttribute('aria-selected', 'false');
  toolsTab.textContent = 'Tools';

  const panel = getOrCreateToolsPanel(tabsRoot);

  toolsTab.addEventListener('click', () => {
    setActiveTab(tabsRoot, toolsTab, panel.id);
    renderToolsHome(panel);
  });

  // Når bruker klikker på annen tab (Metadata/Feedback/…), skjul vårt panel og deaktiver Verktøy
  tabsRoot.addEventListener('click', (e) => {
    const btn = e.target.closest('.tabs__tab');
    if (!btn) return;
    if (btn === toolsTab) return; // vår egen klikk-handling tar seg av dette

    panel.hidden = true;
    toolsTab.classList.remove('tabs__tab--active');
    toolsTab.setAttribute('aria-selected', 'false');
  }, { once: false });

  overflowContent.appendChild(toolsTab);

  // Kjør shrink én gang til etter at Verktøy er lagt til,
  // så den også får samme kompakte stil.
  shrinkEditorTabBar(tabsRoot);
}




// --- Veta kvalitetsjekk kopi-hjelpere ---

function showCopyToast(message) {
  const existing = document.getElementById('veta-copy-toast');
  if (existing) existing.remove();

  const toast = document.createElement('div');
  toast.id = 'veta-copy-toast';
  toast.textContent = message || 'Copied to clipboard ✅';

  Object.assign(toast.style, {
    position: 'fixed',
    bottom: '20px',
    right: '20px',
    padding: '10px 14px',
    backgroundColor: '#333',
    color: '#fff',
    borderRadius: '4px',
    fontSize: '13px',
    zIndex: 9999,
    boxShadow: '0 2px 6px rgba(0,0,0,0.3)',
    opacity: '0',
    transition: 'opacity 0.2s ease'
  });

  document.body.appendChild(toast);
  requestAnimationFrame(() => { toast.style.opacity = '1'; });

  setTimeout(() => {
    toast.style.opacity = '0';
    setTimeout(() => toast.remove(), 200);
  }, 2000);
}

function buildVetaPromptFromPage() {
  // Tittel
  const titleEl = document.querySelector('.text.text--heading-m.mb-1');
  const title = titleEl ? titleEl.textContent.trim() : '(missing title)';

  // Sammendrag – heading-s rett over brødtekst
  const summaryEl = document.querySelector('section.flex.w-full.flex-col .text.text--heading-s.mb-5');
  const summary = summaryEl ? summaryEl.textContent.trim() : '(missing summary)';

  // Brødtekst – alle body-m-avsnitt i samme section
  const bodySection = summaryEl ? summaryEl.closest('section.flex.w-full.flex-col') : null;
  let body = '';

  if (bodySection) {
    const bodyEls = bodySection.querySelectorAll('.text.text--body-m.my-2');
    body = Array.from(bodyEls)
      .map(el => el.textContent.replace(/\s+/g, ' ').trim())
      .filter(Boolean)
      .join('\n\n');
  }

  if (!body) body = '(missing body text)';

  const prompt = `Quality-check this article against Veta guidelines.
Do not include #title, #summary, or #body in the answer; they are for reference only.

#title
-${title}

#summary
-${summary}

#body
-${body}`;

  return prompt;
}

function buildArticleContentFromPage() {
  const titleEl = document.querySelector('.text.text--heading-m.mb-1');
  const title = titleEl ? titleEl.textContent.trim() : '';

  const summaryEl = document.querySelector('section.flex.w-full.flex-col .text.text--heading-s.mb-5');
  const summary = summaryEl ? summaryEl.textContent.trim() : '';

  const bodySection = summaryEl ? summaryEl.closest('section.flex.w-full.flex-col') : null;
  let body = '';

  if (bodySection) {
    const bodyEls = bodySection.querySelectorAll('.text.text--body-m.my-2');
    body = Array.from(bodyEls)
      .map(el => el.textContent.replace(/\s+/g, ' ').trim())
      .filter(Boolean)
      .join('\n\n');
  }

  const parts = [];
  if (title) parts.push(`Title\n${title}`);
  if (summary) parts.push(`Summary\n${summary}`);
  if (body) parts.push(`Body\n${body}`);

  return parts.join('\n\n');
}

function copyTextToClipboard(text) {
  if (navigator.clipboard && navigator.clipboard.writeText) {
    return navigator.clipboard.writeText(text);
  }
  // Fallback
  return new Promise((resolve, reject) => {
    try {
      const ta = document.createElement('textarea');
      ta.value = text;
      ta.style.position = 'fixed';
      ta.style.top = '-9999px';
      document.body.appendChild(ta);
      ta.select();
      document.execCommand('copy');
      document.body.removeChild(ta);
      resolve();
    } catch (e) {
      reject(e);
    }
  });
}


// --- 2.3: Favorittknappen under artikkelbaren ---

function addFavouriteButton() {
  // Respekter settings
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { addFavouriteButton(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const enabled = settings.showFavourites !== false;

  const targetDivs = document.querySelectorAll('.buttons-document.flex.items-center');

  // Hvis AV → fjern eksisterende pin-knapp(er) og stopp
  if (!enabled) {
    targetDivs.forEach(div => {
      const favBtn = div.querySelector('.add-favourite-button');
      if (favBtn) favBtn.remove();
    });
    return;
  }
  targetDivs.forEach(div => {
    // 1) Sørg for at "Pin artikkel"-knappen finnes (som før)
    if (!div.querySelector('.add-favourite-button')) {
      const favouriteButton = document.createElement('button');
      favouriteButton.textContent = 'Pin article';
      favouriteButton.className = 'add-favourite-button';
      favouriteButton.style.marginLeft = '10px';
      favouriteButton.style.padding = '5px 10px';
      favouriteButton.style.backgroundColor = '#f0f0f0';
      favouriteButton.style.border = 'none';
      favouriteButton.style.borderRadius = '4px';
      favouriteButton.style.cursor = 'pointer';
      favouriteButton.style.transition = 'background-color 0.3s';

      favouriteButton.addEventListener('mouseover', () => {
        favouriteButton.style.backgroundColor = '#d0d0d0';
      });
      favouriteButton.addEventListener('mouseout', () => {
        favouriteButton.style.backgroundColor = '#f0f0f0';
      });

      favouriteButton.addEventListener('click', () => {
        const tooltipWrappers = document.querySelectorAll('.tooltip__custom-trigger-wrapper');
        const thirdTooltipWrapper = tooltipWrappers[3];
        const linkElement = thirdTooltipWrapper ? thirdTooltipWrapper.querySelector('a') : null;
        const currentUrl = linkElement ? linkElement.href : window.location.href;

        const titleElement = document.querySelector('.text.text--heading-m.mb-1');
        const defaultTitle = titleElement ? titleElement.textContent.trim() : 'Untitled';

        showTitlePopup(currentUrl, defaultTitle);
      });

      div.appendChild(favouriteButton);
    }

    

    // 2) Legg til vår ny kopiknapp hvis den ikke allerede finnes
    if (!div.querySelector('.veta-copy-button')) {
      const copyBtn = document.createElement('button');
      copyBtn.type = 'button';
      copyBtn.className = 'veta-copy-button';
      copyBtn.setAttribute('aria-label', 'Copy content');
      copyBtn.title = 'Copy content'; // simple hover text

      // Hover tooltip
const tooltip = document.createElement('div');
tooltip.textContent = 'Copy content';
Object.assign(tooltip.style, {
  position: 'absolute',
  background: '#333',
  color: '#fff',
  padding: '4px 8px',
  borderRadius: '4px',
  fontSize: '12px',
  whiteSpace: 'nowrap',
  transform: 'translateY(-120%)',
  opacity: '0',
  pointerEvents: 'none',
  transition: 'opacity 0.2s',
  zIndex: 99999
});
tooltip.style.display = 'none';

copyBtn.style.position = 'relative';
copyBtn.appendChild(tooltip);

copyBtn.addEventListener('mouseover', () => {
  tooltip.style.display = 'block';
  tooltip.style.opacity = '1';
});
copyBtn.addEventListener('mouseout', () => {
  tooltip.style.opacity = '0';
  setTimeout(() => (tooltip.style.display = 'none'), 200);
});


      

      Object.assign(copyBtn.style, {
        marginLeft: '10px',
        padding: '6px 8px',
        backgroundColor: '#f0f0f0',
        border: 'none',
        borderRadius: '4px',
        cursor: 'pointer',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        transition: 'background-color 0.3s'
      });

      copyBtn.addEventListener('mouseover', () => {
        copyBtn.style.backgroundColor = '#d0d0d0';
      });
      copyBtn.addEventListener('mouseout', () => {
        copyBtn.style.backgroundColor = '#f0f0f0';
      });

      // 3 bars – hamburger
      copyBtn.innerHTML = `
       📋
      `;

copyBtn.addEventListener('click', () => {
  try {
    const content = buildArticleContentFromPage();
    copyTextToClipboard(content)
      .then(() => {
        showCopyToast('Copied to clipboard ✅');
      })
      .catch((err) => {
        console.error('Failed to copy:', err);
        showCopyToast('Could not copy 😬');
      });
  } catch (err) {
    console.error('Unexpected error in copy-handler:', err);
    showCopyToast('Something went wrong 😬');
  }
});



      // Plasser kopiknappen til venstre for "Pin artikkel" hvis mulig
      const favouriteButton = div.querySelector('.add-favourite-button');
      if (favouriteButton) {
        div.insertBefore(copyBtn, favouriteButton);
      } else {
        div.appendChild(copyBtn);
      }
    }
  });
}



// --- 2.4: Favoritter i header (med fix mot duplisering) ---

function updateHeaderUtilities() {
  const containers = document.querySelectorAll('.buttons-container-header.flex.items-center');
  if (!containers.length) return;

  // Respekter settings
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { updateHeaderUtilities(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const enabled = settings.showFavourites !== false;

  // Rydd alltid vekk gamle containers først
  containers.forEach(c => {
    const existing = c.querySelector('.saved-favorites-container');
    if (existing) existing.remove();
  });

  // Hvis AV → ikke tegn noe nytt
  if (!enabled) {
    return;
  }

  // ⬅️ Bruk den FØRSTE containeren (den som ligger lengst til venstre, med "Editor"/"Exit editor")
  const targetContainer = containers[0];

  // Rydd vekk gamle favorites-containere i alle
  containers.forEach(c => {
    const existing = c.querySelector('.saved-favorites-container');
    if (existing) existing.remove();
  });

  const favoritesContainer = document.createElement('div');
  favoritesContainer.className = 'saved-favorites-container';
  favoritesContainer.style.display = 'flex';
  favoritesContainer.style.alignItems = 'center';
  favoritesContainer.style.height = '100%';
  favoritesContainer.style.marginRight = '15px';
  favoritesContainer.style.borderRight = '1px solid rgb(255, 255, 255)';

  savedFavorites.forEach((favorite, index) => {
    const favoriteWrapper = document.createElement('div');
    favoriteWrapper.style.position = 'relative';
    favoriteWrapper.style.marginRight = '25px';
    favoriteWrapper.style.display = 'inline-flex';
    favoriteWrapper.style.alignItems = 'center';
    favoriteWrapper.style.height = '100%';

    const favoriteLink = document.createElement('a');
    favoriteLink.href = favorite.url;
    favoriteLink.textContent = `${favorite.title}`;
    favoriteLink.className = 'saved-favorite';
    favoriteLink.style.color = 'black';
    favoriteLink.style.textDecoration = 'none';
    favoriteLink.style.fontSize = '14px';
    favoriteLink.style.position = 'relative';
    favoriteLink.style.cursor = 'pointer';
    favoriteLink.style.display = 'flex';
    favoriteLink.style.alignItems = 'center';
    favoriteLink.style.padding = '12px 20px 10px 10px';
    favoriteLink.style.whiteSpace = 'normal';
    favoriteLink.style.wordWrap = 'break-word';
    favoriteLink.style.overflow = 'hidden';
    favoriteLink.style.textOverflow = 'ellipsis';
    favoriteLink.style.maxWidth = '140px';
    favoriteLink.style.minWidth = '95px';
    favoriteLink.style.maxHeight = "43px";
    favoriteLink.style.boxSizing = 'border-box';
    favoriteLink.style.borderRadius = '5px';
    favoriteLink.style.backgroundColor = 'white';
    favoriteLink.style.justifyContent = 'center';

    const removeButton = document.createElement('button');
    removeButton.textContent = 'x';
    removeButton.style.position = 'absolute';
    removeButton.style.top = '5%';
    removeButton.style.right = '-5px';
    removeButton.style.transform = 'translateY(-50%)';
    removeButton.style.backgroundColor = 'red';
    removeButton.style.color = 'white';
    removeButton.style.border = 'none';
    removeButton.style.borderRadius = '50%';
    removeButton.style.width = '18px';
    removeButton.style.height = '18px';
    removeButton.style.fontSize = '10px';
    removeButton.style.cursor = 'pointer';
    removeButton.style.display = 'flex';
    removeButton.style.justifyContent = 'center';
    removeButton.style.alignItems = 'center';
    removeButton.style.padding = '0';

    removeButton.addEventListener('click', (e) => {
      e.preventDefault();
      e.stopPropagation();
      savedFavorites.splice(index, 1);
      localStorage.setItem('savedFavorites', JSON.stringify(savedFavorites));
      updateHeaderUtilities();
    });

    favoriteWrapper.appendChild(favoriteLink);
    favoriteWrapper.appendChild(removeButton);
    favoritesContainer.appendChild(favoriteWrapper);
  });

  // Sett favorites inn til venstre, resten (inkl. Editor-knapp) skyves til høyre
  targetContainer.insertBefore(favoritesContainer, targetContainer.firstChild);

  targetContainer.style.display = 'flex';
  targetContainer.style.justifyContent = 'flex-start';
  targetContainer.style.width = '100%';

  Array.from(targetContainer.children).forEach(child => {
    if (child !== favoritesContainer) child.style.marginLeft = 'auto';
  });
}

function showTitlePopup(url, defaultTitle) {
  const popup = document.createElement('div');
  popup.style.position = 'fixed';
  popup.style.top = '50%';
  popup.style.left = '50%';
  popup.style.transform = 'translate(-50%, -50%)';
  popup.style.backgroundColor = 'white';
  popup.style.padding = '20px';
  popup.style.borderRadius = '5px';
  popup.style.boxShadow = '0 2px 10px rgba(0, 0, 0, 0.2)';
  popup.style.zIndex = '9999';

  const titleInput = document.createElement('input');
  titleInput.type = 'text';
  titleInput.value = defaultTitle;
  titleInput.style.width = '100%';
  titleInput.style.marginBottom = '10px';
  titleInput.style.padding = '5px';
  titleInput.style.backgroundColor = '#f2f2f2';

  const saveButton = document.createElement('button');
  saveButton.textContent = 'Save';
  saveButton.style.marginRight = '10px';
  saveButton.style.padding = '5px 10px';
  saveButton.style.backgroundColor = '#4CAF50';
  saveButton.style.color = 'white';
  saveButton.style.border = 'none';
  saveButton.style.borderRadius = '3px';
  saveButton.style.cursor = 'pointer';

  const cancelButton = document.createElement('button');
  cancelButton.textContent = 'Cancel';
  cancelButton.style.padding = '5px 10px';
  cancelButton.style.backgroundColor = '#f44336';
  cancelButton.style.color = 'white';
  cancelButton.style.border = 'none';
  cancelButton.style.borderRadius = '3px';
  cancelButton.style.cursor = 'pointer';

  popup.appendChild(titleInput);
  popup.appendChild(saveButton);
  popup.appendChild(cancelButton);
  document.body.appendChild(popup);

  saveButton.addEventListener('click', () => {
    const customTitle = titleInput.value.trim();
    savedFavorites.push({ url: url, title: customTitle });
    localStorage.setItem('savedFavorites', JSON.stringify(savedFavorites));
    updateHeaderUtilities();
    document.body.removeChild(popup);
  });

  cancelButton.addEventListener('click', () => {
    document.body.removeChild(popup);
  });
}

function logFlexColElements() {
  const flexColElements = document.querySelectorAll('#tools');
  console.log('Elements with id #tools:');
  flexColElements.forEach((element, index) => {
    console.log(`Element ${index + 1}:`, element);
  });
  console.log(`Total number of elements with id #tools: ${flexColElements.length}`);
}

async function runVetaQualityCheck() {
  const sleep = (ms) => new Promise(res => setTimeout(res, ms));

  function getRows() {
    return Array.from(document.querySelectorAll('.grid.cursor-pointer'));
  }

  function pickRandomRow(rows) {
    const idx = Math.floor(Math.random() * rows.length);
    return rows[idx];
  }

  async function waitForModal(timeoutMs = 6000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const modalBody = document.querySelector('.sheets__content-wrapper .modal-body');
      if (modalBody) return modalBody;
      await sleep(100);
    }
    return null;
  }

  function closeModal() {
    const backdrop = document.querySelector('.modal-wrapper__backdrop');
    if (backdrop) backdrop.click();
  }

  function extractSummaryAndBody(modal) {
    let contentSection = null;
    const sections = Array.from(modal.querySelectorAll('section.flex.w-full.flex-col'));

    contentSection =
      sections.find(sec => sec.querySelector('.buttons-document')) ||
      sections[sections.length - 1];

    if (!contentSection) {
      console.warn('❌ Could not find contentSection');
      return { summary: '', body: '' };
    }

    const clone = contentSection.cloneNode(true);

    const summaryEl = clone.querySelector('.text.text--heading-s');
    const summary = summaryEl ? summaryEl.innerText.trim() : '';
    if (summaryEl) summaryEl.remove();

    const buttons = clone.querySelector('.buttons-document');
    if (buttons) buttons.remove();

    function nodeToText(node) {
      if (node.nodeType === Node.TEXT_NODE) {
        return node.nodeValue;
      }
      if (node.nodeType !== Node.ELEMENT_NODE) {
        return '';
      }

      const tag = node.tagName.toLowerCase();
      const childrenText = Array.from(node.childNodes).map(nodeToText).join('');

      switch (tag) {
        case 'br':
          return '\n';
        case 'li':
          return '\n- ' + childrenText.trim();
        case 'ul':
        case 'ol':
          return '\n' + childrenText.trim() + '\n';
        case 'p':
        case 'div':
        case 'section':
          return '\n' + childrenText.trim() + '\n';
        default:
          return childrenText;
      }
    }

    let bodyText = nodeToText(clone) || '';

    bodyText = bodyText
      .replace(/\u00a0/g, ' ')
      .replace(/[ \t]+\n/g, '\n')
      .replace(/\n{3,}/g, '\n\n')
      .trim();

    return { summary, body: bodyText };
  }

  function buildUrlForArticle(id, title) {
    const meta = window.__vetaLastDocMeta;
    if (!meta || !meta.guid) return '';

    // Litt “best effort”-matching for å unngå feil guid
    if (meta.vetaDocId && id && String(meta.vetaDocId) !== String(id)) {
      if (meta.title && title && meta.title.trim() !== title.trim()) {
        return '';
      }
    }

    const market = (meta.marketCode || 'NO').toLowerCase();
    return `https://km.ingka.com/document/${meta.guid}?lang=${market}&market=${market}`;
  }

  function extractArticle(modal) {
    const id = modal.querySelector('.document-card-header .text-sm.text-gray-400')?.innerText?.trim() || '';
    const title = modal.querySelector('.text.text--heading-m[role="heading"]')?.innerText?.trim() || '';

    const { summary, body } = extractSummaryAndBody(modal);
    const url = buildUrlForArticle(id, title);

    return { id, title, summary, body, url };
  }

  async function openAndExtract(row, index) {
    // Nullstill før vi åpner ny artikkel
    window.__vetaLastDocMeta = null;

    row.click();

    const modal = await waitForModal(6000);
    if (!modal) {
      console.warn('❌ Could not find modal for this article');
      return null;
    }

    // Gi fetch + botAnswer-logger litt tid til å sette __vetaLastDocMeta
    await sleep(500);

    const article = extractArticle(modal);
    console.log('✅ Fetched article:', article);

    closeModal();
    await sleep(400);

    return article;
  }

  const rows = getRows();
  if (!rows.length) {
    console.warn('No article rows found (.grid.cursor-pointer)');
    showCopyToast('No articles found in the list 😬');
    return;
  }

  const TARGET_COUNT = 2; // hvor mange artikler du vil sample per klikk
  const articles = [];
  const usedIds = new Set();
  let safety = 0;
  const maxSafety = rows.length * 10;

  while (articles.length < TARGET_COUNT && safety < maxSafety) {
    safety++;
    const row = pickRandomRow(rows);
    const art = await openAndExtract(row, articles.length + 1);
    if (!art || !art.id) continue;

    if (usedIds.has(art.id)) {
      continue;
    }

    usedIds.add(art.id);
    articles.push(art);
  }

  if (!articles.length) {
    console.warn('No articles were collected.');
    showCopyToast('No articles to send for quality check 😬');
    return;
  }

  const promptHeader = `
You are a senior VETA content reviewer.

You receive a JSON object with a list of articles from VETA. Each article has the fields:
- id: VETA document ID
- title: title in VETA
- summary: short ingress / summary text (if present)
- body: main content of the article (plain text)
- url: direct link to the article in VETA (if available)

Your task is to evaluate each article against the criteria described in "Veta kriterier.md" - with these important clarifications:

IMPORTANT - internal sections at the bottom ("Internal information (only visible to co-workers)")
- This is normal in VETA and should not be penalized.
- This is a separate information layer for employees and does not affect the target audience evaluation.
- IGNORE this part in the assessment unless it contains clear errors.

IMPORTANT - FAQ / short articles
- Short FAQ articles do not need subheadings or bullet lists.
- Do not penalize "lack of scannability" if the article is short enough that bullet lists would not add value.
- Do not penalize "missing links" unless the article clearly should help the user move on to a more complex topic.

IMPORTANT - links
- It is OK for articles to have no links.
- Links should be relevant and clearly named (not just "here"), but only require links in longer or more complex articles.

-------------------------------------------------------------

OUTPUT format:

For each article, provide:

1. HEADING
"Article {id}: {title}"

2. URL (if available)
Write:
"URL: {url}"
If url is missing or empty, you may omit this line.

3. SHORT SUMMARY
1-2 sentences about what the article is about.

-------------------------------------------------------------

3. CONTENT QUALITY
Create 4-7 bullet lines.

- Each line must start with an emoji:
  - "✅" if the criterion is met
  - "⚠️" if it is partially met
  - "❌" if it is not met

- Format for each line:
  "⚠️ Links: Partially - useful link, but could have had more relevant options"
  "❌ Scannability: No - long paragraphs without structure"
  "✅ Uniqueness: Yes - clearly scoped topic"

- End the section with:
  Score (Content Quality): choose one of "✅", "⚠️", or "❌".

-------------------------------------------------------------

4. IKEA TONE OF VOICE
Same structure:

- Bullet lines with emoji first
- Short label + "Yes/Partly/No - explanation"
- End with:
  Score (Tone of Voice): "✅" / "⚠️" / "❌"

-------------------------------------------------------------

5. DOCUMENT STRUCTURE
Same structure, but with these rules:

- Do not penalize missing bullet lists in short articles.
- Do not penalize the internal section at the bottom. This is normal.
- Do not require a question title if the article is not an FAQ.

End with:
  Score (Document Structure): "✅" / "⚠️" / "❌"

-------------------------------------------------------------

6. IMPROVEMENT SUGGESTIONS
Give 3-8 concrete suggestions.
Use "-" or "•" as bullet markers (no emoji needed).

-------------------------------------------------------------

7. OVERLAP (only if relevant)
If two or more articles clearly overlap:

- Add a section:
  "Possible overlap / duplicate:"
- List the articles (id + title) and briefly explain how they overlap.

-------------------------------------------------------------

IMPORTANT:
Never penalize "Internal information (only visible to co-workers)". This section must be ignored.
`.trim();

  const payload = {
    generatedAt: new Date().toISOString(),
    articles
  };

  const textToCopy = `${promptHeader}\n\nJSON:\n${JSON.stringify(payload, null, 2)}`;

 


  await copyTextToClipboard(textToCopy);
  showCopyToast('Copied! Paste into MAP KS specialist 👇');
  console.log('VETA quality payload:\n', textToCopy);

  // Open MAP in a new tab
    window.open(
      "https://ai.ingka.com/chat?a=d7d4b570-6633-4918-ba42-a2535e4ff49f",
      "_blank"
    );
}

async function exportAllVetaDocsToJson() {
  const sleep = (ms) => new Promise(res => setTimeout(res, ms));

  // --- 1. Hook into __vetaPrettyLog to capture GetDoc ---
  const collectedByGuid = {};
  const originalPrettyLog = window.__vetaPrettyLog;

  window.__vetaPrettyLog = function (payload) {
    // Preserve existing behavior (botAnswer box, etc.)
    try {
      if (typeof originalPrettyLog === "function") {
        originalPrettyLog(payload);
      }
    } catch (e) {
      console.warn("Original __vetaPrettyLog failed:", e);
    }

    // Plukk ut dokumentdata
    try {
      const doc = payload && payload.data && payload.data.GetDoc;
      if (!doc || !doc.guid) return;

      collectedByGuid[doc.guid] = {
        guid: doc.guid,
        id: String(doc.vetaDocId || doc.id || ""),
        title: doc.title || "",
        keywords: Array.isArray(doc.keywords) ? doc.keywords.filter(Boolean) : [],
        marketCode: doc.marketCode || ""
      };
    } catch (e) {
      console.warn("Could not collect doc from payload:", e);
    }
  };

  // --- 2. Helpers for list/“Show more”/modal ---

  async function loadAllRows() {
    let clicks = 0;
    let lastCount = 0;
    let sameCountStreak = 0;

    const countRows = () => document.querySelectorAll(".grid.cursor-pointer").length;

    while (true) {
      const btn = findVetaLoadMoreButton();
      if (!btn || btn.offsetParent === null) {
        console.log("No more load-more buttons – stop.");
        break;
      }

      const before = countRows();
      btn.click();
      clicks++;

      console.log(`Click #${clicks} on load-more button...`);

      let waited = 0;
      let grew = false;
      for (;;) {
        await sleep(200);
        waited += 200;
        const now = countRows();
        if (now > before) {
          grew = true;
          break;
        }
        if (waited >= 5000) break;
      }

      const currentCount = countRows();
      if (currentCount === lastCount) sameCountStreak++;
      else { sameCountStreak = 0; lastCount = currentCount; }

      if (!grew && sameCountStreak >= 2) {
        console.log("Rows are no longer increasing – stopping.");
        break;
      }
      if (clicks >= 500) {
        console.warn("Reached 500 load-more clicks – stopping for safety.");
        break;
      }
    }
  }

  function getRows() {
    return Array.from(document.querySelectorAll(".grid.cursor-pointer"));
  }

  async function waitForModal(timeoutMs = 6000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const modalBody = document.querySelector(".sheets__content-wrapper .modal-body");
      if (modalBody) return modalBody;
      await sleep(100);
    }
    return null;
  }

  function closeModal() {
    const backdrop = document.querySelector(".modal-wrapper__backdrop");
    if (backdrop) backdrop.click();
  }

  async function openRowAndWait(row, index, total) {
    console.log(`Opening row ${index + 1} / ${total}...`);
    row.click();

    const modal = await waitForModal(7000);
    if (!modal) {
      console.warn("Did not find modal for row", index + 1);
      return false;
    }

    // gi fetch / GraphQL litt tid
    await sleep(600);

    // lukk igjen
    closeModal();
    await sleep(300);

    return true;
  }

  // --- 3. Kjør hele greia ---

  console.log("🔄 Loading all rows via load-more button...");
  await loadAllRows();

  const rows = getRows();
  if (!rows.length) {
    console.warn("No .grid.cursor-pointer rows found. Are you on the /document list?");
    // restore hook
    window.__vetaPrettyLog = originalPrettyLog;
    return;
  }

  console.log(`✅ Found ${rows.length} rows. Opening each to collect ID, title and keywords...`);

  for (let i = 0; i < rows.length; i++) {
    await openRowAndWait(rows[i], i, rows.length);
  }

  // --- 4. Ferdig – fjern hook, lag JSON og last ned ---

  window.__vetaPrettyLog = originalPrettyLog;

  const docs = Object.values(collectedByGuid);
  console.log(`🎉 Done! Collected ${docs.length} documents (unique by guid).`);

  const json = JSON.stringify(docs, null, 2);
  console.log("Example entry:", docs[0] || "(no docs)");

  try {
    const blob = new Blob([json], { type: "application/json" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "veta-docs.json";
    document.body.appendChild(a);
    a.click();
    setTimeout(() => {
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    }, 0);
    console.log("💾 Started download of veta-docs.json");
    if (window.showCopyToast) {
      window.showCopyToast("JSON export started – file download should begin shortly ✅");
    }
  } catch (e) {
    console.warn("Could not trigger download – dumping JSON in console instead.");
    console.log(json);
  }

  return docs;
}




/* Spinner + Ratio checked  */

function ensureSpinnerStyles() {
  if (document.getElementById('veta-spinner-style')) return;

  const style = document.createElement('style');
  style.id = 'veta-spinner-style';
  style.textContent = `
    @keyframes veta-spin {
      to { transform: rotate(360deg); }
    }
    .veta-spinner {
      display: inline-block;
      width: 14px;
      height: 14px;
      border-radius: 50%;
      border: 2px solid currentColor;
      border-right-color: transparent;
      border-bottom-color: transparent;
      animation: veta-spin 0.7s linear infinite;
      vertical-align: -2px;
    }
  `;
  document.head.appendChild(style);

}

// Finner "Vis mer" / "Visa mer" / "Show more"-knappen uavhengig av språk
function findVetaLoadMoreButton() {
  // Primært: den knappen du viste – button inni .py-2.text-center
  const containerButtons = Array.from(
    document.querySelectorAll('.py-2.text-center button')
  );
  const visible = containerButtons.find(b => b.offsetParent !== null);
  if (visible) return visible;

  // Fallback: liten sekundær-knapp (i tilfelle layout endres litt)
  const fallbackButtons = Array.from(
    document.querySelectorAll('button.btn.btn--small.btn--secondary')
  );
  return fallbackButtons.find(b => b.offsetParent !== null) || null;
}


async function runNoEnRatioCheck() {
  const MAX_CLICKS = 500;
  const XLSX_VERSION = '0.18.5';
  const wait = (ms) => new Promise(r => setTimeout(r, ms));

  async function loadAllRows() {
    let clicks = 0;
    let lastCount = 0;
    let sameCountStreak = 0;
    const countRows = () => document.querySelectorAll('.product-identifier__value').length;

    while (true) {
      const btn = findVetaLoadMoreButton();
      if (!btn || btn.offsetParent === null) break;

      btn.click();
      clicks++;

      let waited = 0;
      let grew = false;
      const start = countRows();
      for (;;) {
        await wait(150);
        waited += 150;
        const now = countRows();
        if (now > start) { grew = true; break; }
        if (waited >= 5000) break;
      }

      const currentCount = countRows();
      if (currentCount === lastCount) sameCountStreak++;
      else { sameCountStreak = 0; lastCount = currentCount; }

      if (!grew && sameCountStreak >= 2) break;
      if (clicks >= MAX_CLICKS) {
        console.warn('Reached MAX_CLICKS - stopping for safety.');
        break;
      }
    }
  }


  const isRedBorder = (el) => {
    if (!el) return false;
    const cs = getComputedStyle(el);
    const w = parseFloat(cs.borderTopWidth) || 0;
    const style = cs.borderTopStyle;
    const color = cs.borderTopColor;
    const looksRed = color === 'rgb(255, 0, 0)' || color === 'red';
    return w > 0 && style !== 'none' && looksRed;
  };

  async function ensureSheetJS() {
    if (window.XLSX) return true;
    return new Promise(resolve => {
      const s = document.createElement('script');
      s.src = `https://cdn.jsdelivr.net/npm/xlsx@${XLSX_VERSION}/dist/xlsx.full.min.js`;
      s.onload = () => resolve(true);
      s.onerror = () => resolve(false);
      document.head.appendChild(s);
    });
  }

  function collectData() {
    const enBadges = Array.from(document.querySelectorAll('.product-identifier__value'))
      .filter(el => (el.textContent || '').trim().toLowerCase() === 'en');

    const rows = enBadges.map(valueEl => {
      const badge = valueEl.closest('.product-identifier');
      const row = badge?.closest('.grid') || null;

      const titleDiv = row?.querySelector('.font-medium');
      let idText = '';
      let titleText = '';
      if (titleDiv) {
        const lines = (titleDiv.innerText || '').split('\n').map(s => s.trim()).filter(Boolean);
        idText = lines[0] || '';
        titleText = lines[1] || '';
      }

      const notTranslated = isRedBorder(badge);
      const status = notTranslated ? 'Ikke oversatt' : 'Oversatt';

      return { id: idText, title: titleText, status };
    });

    // Dedup på id+title
    const dedup = [];
    const seen = new Set();
    for (const r of rows) {
      const key = `${r.id}||${r.title}`;
      if (!seen.has(key)) { seen.add(key); dedup.push(r); }
    }
    return dedup;
  }

  const pct = (n, total) => total ? ((n/total)*100).toFixed(1) + '%' : '0.0%';

  // === Kjør ===
  console.log('Loading all rows by clicking "Show more"...');
  await loadAllRows();

  const data = collectData();
  const total = data.length;
  const translated = data.filter(d => d.status === 'Oversatt').length;
  const notTranslated = total - translated;

  // Last SheetJS (må ha for XLSX)
  const ok = await ensureSheetJS();
  if (!ok || !window.XLSX) {
    console.error('Failed to load SheetJS. Cannot create XLSX.');
    return;
  }

  // ------- XLSX: Oversikt-fane -------
  const today = new Date().toISOString().slice(0,10);
  const topLine = [
    `Totalt: ${total} | Oversatt: ${translated} (${pct(translated, total)}) | ` +
    `Ikke oversatt: ${notTranslated} (${pct(notTranslated, total)})`
  ];
  const overviewRows = [
    [`Oversikt per ${today}`],
    [''],
    topLine,
    [''],
    ['(index)','Metric','Antall','Andel'],
    ['0','Totalt', total, pct(total, total)],
    ['1','Oversatt', translated, pct(translated, total)],
    ['2','Ikke oversatt', notTranslated, pct(notTranslated, total)],
  ];
  const wsOverview = XLSX.utils.aoa_to_sheet(overviewRows);
  wsOverview['!cols'] = [{wch:10},{wch:16},{wch:10},{wch:10}];

  // ------- XLSX: Artikler-fane -------
  const articleRows = [
    ['ID','Tittel','Status'],
    ...data.map(r => [r.id, r.title, r.status])
  ];
  const wsArticles = XLSX.utils.aoa_to_sheet(articleRows);
  wsArticles['!cols'] = [{wch:12},{wch:80},{wch:14}];

  // ------- Workbook -------
  const wb = XLSX.utils.book_new();
  XLSX.utils.book_append_sheet(wb, wsOverview, 'Oversikt');
  XLSX.utils.book_append_sheet(wb, wsArticles, 'Artikler');
  XLSX.writeFile(wb, `articles_${today}.xlsx`);

  console.log('===== Oversettelsesstatus =====');
  console.log(topLine[0]);
}

function showJsonExportConfirmDialog() {
  return new Promise((resolve) => {
    const overlay = document.createElement('div');
    Object.assign(overlay.style, {
      position: 'fixed',
      inset: 0,
      background: 'rgba(0,0,0,0.35)',
      display: 'flex',
      alignItems: 'center',
      justifyContent: 'center',
      zIndex: 9999
    });

    const box = document.createElement('div');
    Object.assign(box.style, {
      background: '#fff',
      borderRadius: '6px',
      padding: '16px 18px',
      maxWidth: '420px',
      width: '100%',
      boxShadow: '0 4px 18px rgba(0,0,0,0.25)',
      fontFamily: 'system-ui, -apple-system, Segoe UI, sans-serif',
      fontSize: '14px'
    });

    const header = document.createElement('div');
    header.textContent = 'JSON export';
    Object.assign(header.style, {
      fontWeight: '600',
      marginBottom: '8px'
    });

    const msg = document.createElement('div');
    msg.textContent = 'Notice: The JSON export may require several minutes to finish. Proceed only if you have a defined business need for the output file, as the operation performs a comprehensive scan through all available documents.';
    Object.assign(msg.style, {
      marginBottom: '12px',
      lineHeight: '1.5'
    });

    const buttonsRow = document.createElement('div');
    Object.assign(buttonsRow.style, {
      display: 'flex',
      justifyContent: 'flex-end',
      gap: '8px',
      marginTop: '4px'
    });

    const cancelBtn = document.createElement('button');
    cancelBtn.textContent = 'Cancel';
    Object.assign(cancelBtn.style, {
      padding: '6px 10px',
      borderRadius: '4px',
      border: '1px solid #ccc',
      background: '#f5f5f5',
      cursor: 'pointer',
      fontSize: '13px'
    });

    const runBtn = document.createElement('button');
    runBtn.textContent = 'Run';
    Object.assign(runBtn.style, {
      padding: '6px 12px',
      borderRadius: '4px',
      border: 'none',
      background: '#0058a3',
      color: '#fff',
      cursor: 'pointer',
      fontSize: '13px'
    });

    function cleanup(result) {
      document.body.removeChild(overlay);
      resolve(result);
    }

    cancelBtn.addEventListener('click', () => cleanup(false));
    runBtn.addEventListener('click', () => cleanup(true));

    // Lukk ved klikk utenfor boksen
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) cleanup(false);
    });

    buttonsRow.appendChild(cancelBtn);
    buttonsRow.appendChild(runBtn);
    box.appendChild(header);
    box.appendChild(msg);
    box.appendChild(buttonsRow);
    overlay.appendChild(box);
    document.body.appendChild(overlay);
  });
}


function ensureNoEnRatioButton() {
  // Extra tools are now provided by the site; disable extension menu.
  return;
  // Only on /document list page
  if (!location.pathname.startsWith('/document')) return;

  // Make sure settings are loaded before we decide what to show
  if (!window.__vetaSettingsLoaded) {
    loadVetaSettings(() => {
      try { ensureNoEnRatioButton(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;

  const showMenu    = settings.showExtraMenu !== false;
  const showRatio   = settings.showExtraRatio !== false;
  const showQuality = settings.showExtraQuality !== false;
  const showJson    = settings.showExtraJson !== false;

  const anyToolOn = showRatio || showQuality || showJson;

  const bar = document.querySelector('.mb-4.flex.space-x-2');
  if (!bar) return;

  // Look up existing elements if present
  let extraBtn = bar.querySelector('.veta-extra-btn');
  let menu     = bar.querySelector('.veta-extra-menu');

  // If menu is disabled OR no tools enabled → remove Extra UI and bail out
  if (!showMenu || !anyToolOn) {
    if (extraBtn) extraBtn.remove();
    if (menu) menu.remove();
    return;
  }

  ensureSpinnerStyles();

  // Create Extra button if needed
  if (!extraBtn) {
    extraBtn = document.createElement('button');
    extraBtn.type = 'button';
    extraBtn.className = 'btn btn--xsmall btn--secondary veta-extra-btn';
    extraBtn.innerHTML = '<span class="btn__inner">Extra</span>';
    extraBtn.style.marginLeft = '8px';
    bar.appendChild(extraBtn);
  }

  // Create menu container if needed
  if (!menu) {
    menu = document.createElement('div');
    menu.className = 'veta-extra-menu';
    Object.assign(menu.style, {
      display: 'none',
      alignItems: 'center',
      gap: '8px'
    });
    bar.appendChild(menu);
  }

  // --- Helper to create/remove individual tool buttons -----------------

  function ensureToolButton(opts) {
    const { className, label, onClick, enabled } = opts;
    let btn = menu.querySelector('.' + className);

    if (!enabled) {
      if (btn) btn.remove();
      return;
    }

    if (!btn) {
      btn = document.createElement('button');
      btn.type = 'button';
      btn.className = 'btn btn--xsmall btn--secondary ' + className;
      btn.innerHTML = `<span class="btn__inner">${label}</span>`;
      menu.appendChild(btn);
    }

    if (!btn.__vetaInit) {
      btn.__vetaInit = true;
      onClick(btn);
    }
  }

  // --- Ratio button ----------------------------------------------------

  ensureToolButton({
    className: 'no-en-ratio-btn',
    label: 'No / EN ratio check',
    enabled: showRatio,
    onClick: (btn) => {
      btn.addEventListener('click', () => {
        if (btn.__running) return;

        btn.__running = true;
        const originalInner = btn.innerHTML;
        btn.disabled = true;

        btn.innerHTML = `
          <span class="btn__inner">
            <span class="veta-spinner" aria-hidden="true"></span>
            <span style="margin-left: 6px;">Running…</span>
          </span>
        `;

        (async () => {
          try {
            await runNoEnRatioCheck();
          } catch (err) {
            console.error('Feil i No/EN ratio-scriptet:', err);
          } finally {
            btn.disabled = false;
            btn.__running = false;
            btn.innerHTML = originalInner;
          }
        })();
      });
    }
  });

  // --- Quality check button -------------------------------------------

  ensureToolButton({
    className: 'veta-quality-btn',
    label: 'Quality check',
    enabled: showQuality,
    onClick: (btn) => {
      btn.addEventListener('click', () => {
        if (btn.__running) return;

        btn.__running = true;
        const originalInner = btn.innerHTML;
        btn.disabled = true;

        btn.innerHTML = `
          <span class="btn__inner">
            <span class="veta-spinner" aria-hidden="true"></span>
            <span style="margin-left: 6px;">Running…</span>
          </span>
        `;

        (async () => {
          try {
            await runVetaQualityCheck();
          } catch (err) {
            console.error('Feil i VETA quality check-scriptet:', err);
          } finally {
            btn.disabled = false;
            btn.__running = false;
            btn.innerHTML = originalInner;
          }
        })();
      });
    }
  });

  // --- JSON export button ---------------------------------------------

  ensureToolButton({
    className: 'veta-json-btn',
    label: 'JSON export',
    enabled: showJson,
    onClick: (btn) => {
      btn.addEventListener('click', async () => {
        if (btn.__running) return;

        const shouldRun = await showJsonExportConfirmDialog();
        if (!shouldRun) {
          console.log('JSON export cancelled by user.');
          return;
        }

        btn.__running = true;
        const originalInner = btn.innerHTML;
        btn.disabled = true;

        btn.innerHTML = `
          <span class="btn__inner">
            <span class="veta-spinner" aria-hidden="true"></span>
            <span style="margin-left: 6px;">Running…</span>
          </span>
        `;

        (async () => {
          try {
            await exportAllVetaDocsToJson();
          } catch (err) {
            console.error('Error in JSON export script:', err);
            if (window.showCopyToast) {
              window.showCopyToast('JSON export failed 😬');
            }
          } finally {
            btn.disabled = false;
            btn.__running = false;
            btn.innerHTML = originalInner;
          }
        })();
      });
    }
  });

  // Toggle menu when Extra is clicked
  if (!extraBtn.__vetaExtraInit) {
    extraBtn.__vetaExtraInit = true;
    extraBtn.addEventListener('click', () => {
      const menuEl = bar.querySelector('.veta-extra-menu');
      if (!menuEl) return;
      const visible = menuEl.style.display === 'flex';
      menuEl.style.display = visible ? 'none' : 'flex';
    });
  }
}




/* ===========================
   Extension Settings Menu Item
   =========================== */

function injectExtensionSettingsItem() {
  // This element exists only when the drawer is open
  const sections = document.querySelector('.cwds-user-profile__sections');
  if (!sections) return;

  // Avoid duplicates
  if (sections.querySelector('.veta-extension-settings-item')) return;

  // Find an existing row to clone
  const sample = sections.querySelector('.cwds-user-profile__section-item');
  if (!sample) return;

  const item = sample.cloneNode(true);
  item.classList.add('veta-extension-settings-item');

  // Update text
  const titleEl = item.querySelector('.cwds-user-profile__title');
  const descEl = item.querySelector('.cwds-user-profile__description');

  if (titleEl) titleEl.textContent = 'Extension settings';
  if (descEl) descEl.textContent = 'Configure the Veta extension';

  // Remove any onclick or framework crap
  item.replaceWith(item.cloneNode(true));
  
  // Add click handler (we will replace this with real modal later)
  item.addEventListener('click', (e) => {
    e.stopPropagation();
    e.preventDefault();
    openVetaExtensionSettings();
  });

  sections.appendChild(item);
}

function openVetaExtensionSettings() {
  loadVetaSettings((settings) => {
    let overlay = document.getElementById('veta-extension-settings-overlay');

    if (!overlay) {
      overlay = document.createElement('div');
      overlay.id = 'veta-extension-settings-overlay';
      Object.assign(overlay.style, {
        position: 'fixed',
        inset: '0',
        background: 'rgba(0,0,0,0.35)',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        zIndex: 9999
      });
      document.body.appendChild(overlay);
    }

    overlay.style.display = 'flex';

    overlay.innerHTML = `
      <div class="veta-settings-panel" style="
        background:#fff;
        border-radius:10px;
        max-width:600px;
        width:95%;
        max-height:80vh;
        padding:16px 18px 18px;
        box-shadow:0 10px 40px rgba(0,0,0,0.25);
        font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;
        font-size:14px;
        display:flex;
        flex-direction:column;
        gap:12px;
      ">
        <div style="display:flex; align-items:center; justify-content:space-between; gap:8px;">
          <div style="font-size:18px; font-weight:600;">Veta extension settings</div>

          <div style="display:flex; align-items:center; gap:12px;">
            <a href="#" target="_blank" 
               id="veta-settings-learn-more"
               style="
                 font-size:13px;
                 color:#0058a3;
                 text-decoration:none;
                 cursor:pointer;
               ">
              Learn more
            </a>

          
          </div>
        </div>



        <div style="font-size:13px; color:#555;">
          Configure which tools, visuals, and theme the Veta helper extension should use.
        </div>

        <div style="border-top:1px solid #eee; margin:4px 0 0;"></div>

        <!-- Theme section -->
        <div style="display:flex; flex-direction:column; gap:10px; margin-top:8px;">
          <div style="font-weight:600; font-size:13px; text-transform:uppercase; letter-spacing:0.03em; color:#666;">
            Theme
          </div>

          <div style="display:flex; flex-wrap:wrap; gap:10px; font-size:13px;">
            <label style="display:flex; align-items:center; gap:6px; cursor:pointer;">
              <input type="radio" name="veta-theme" value="default" id="veta-theme-default">
              <span>Default</span>
            </label>

            <label style="display:flex; align-items:center; gap:6px; cursor:pointer;">
              <input type="radio" name="veta-theme" value="pink" id="veta-theme-pink">
              <span>Cozy pink</span>
            </label>

            <label style="display:flex; align-items:center; gap:6px; cursor:pointer;">
              <input type="radio" name="veta-theme" value="ikea" id="veta-theme-ikea">
              <span>IKEA</span>
            </label>
          </div>
        </div>

        <!-- Article view & editor -->
        <div style="display:flex; flex-direction:column; gap:10px; margin-top:8px;">
          <div style="font-weight:600; font-size:13px; text-transform:uppercase; letter-spacing:0.03em; color:#666;">
            Article view & editor
          </div>

          <label style="display:flex; align-items:flex-start; gap:8px; cursor:pointer;">
            <input type="checkbox"
                   id="veta-setting-show-botanswer-box"
                   style="margin-top:3px;">
            <div>
              <div style="font-weight:500;">Show BotAnswer box</div>
              <div style="font-size:12px; color:#666;">
                Shows the BotAnswer preview box below the article header.
              </div>
            </div>
          </label>

          <label style="display:flex; align-items:flex-start; gap:8px; cursor:pointer;">
            <input type="checkbox"
                   id="veta-setting-show-favourites"
                   style="margin-top:3px;">
            <div>
              <div style="font-weight:500;">Show favourites (Pin article)</div>
              <div style="font-size:12px; color:#666;">
                Enables the “Pin article” button and favourites links in the top header.
              </div>
            </div>
          </label>
        </div>

        <div style="border-top:1px solid #eee; margin:10px 0 0;"></div>

       

          <label style="display:flex; align-items:flex-start; gap:8px; cursor:pointer;">
            <input type="checkbox"
                   id="veta-setting-show-chatbot-preview"
                   style="margin-top:3px;">
            <div>
              <div style="font-weight:500;">Enable chatbot preview</div>
              <div style="font-size:12px; color:#666;">
                Shows the “Show as chatbot” button in the BotAnswer box.
              </div>
            </div>
          </label>

          <label style="display:flex; align-items:flex-start; gap:8px; cursor:pointer;">
            <input type="checkbox"
                   id="veta-setting-show-chatbot-articles"
                   style="margin-top:3px;">
            <div>
              <div style="font-weight:500;">Show “Chatbot articles” menu</div>
              <div style="font-size:12px; color:#666;">
                Adds the Chatbot articles entry in the editor left-hand menu.
              </div>
            </div>
          </label>

          <div style="display:flex; align-items:flex-start; gap:8px; margin-top:4px;">
            <div>
              <div style="font-weight:500;">Profile icon</div>
              <div style="font-size:12px; color:#666;">
                Change profile icon to an emoji (e.g. 🙂). Leave empty for default icon.
              </div>
              <input type="text"
                     id="veta-setting-chatbot-icon"
                     maxlength="8"
                     style="margin-top:4px; width:80px; padding:4px 6px; border-radius:4px; border:1px solid #ccc; font-size:14px; position:relative; z-index:2; pointer-events:auto;">
            </div>
          </div>

           <div style="display:flex; justify-content:flex-end; gap:8px; margin-top:10px;">
          <button type="button" class="veta-settings-close" style="
            padding:6px 10px;
            border-radius:6px;
            border:1px solid #ccc;
            background:#f5f5f5;
            cursor:pointer;
            font-size:13px;
          ">Close</button>
        </div>

      </div>
    `;

    // --- Hook up controls ---

    const showExtraMenu      = overlay.querySelector('#veta-setting-show-extra-menu');
    const showExtraRatio     = overlay.querySelector('#veta-setting-show-extra-ratio');
    const showExtraQuality   = overlay.querySelector('#veta-setting-show-extra-quality');
    const showExtraJson      = overlay.querySelector('#veta-setting-show-extra-json');

    const showBotAnswerBox   = overlay.querySelector('#veta-setting-show-botanswer-box');
    const showEditorTools    = overlay.querySelector('#veta-setting-show-editor-tools');
    const showFavourites     = overlay.querySelector('#veta-setting-show-favourites');

     const showChatbotPreview = overlay.querySelector('#veta-setting-show-chatbot-preview');
    const showChatbotArticles = overlay.querySelector('#veta-setting-show-chatbot-articles');
        const chatbotIconInput   = overlay.querySelector('#veta-setting-chatbot-icon');



    const themeDefault = overlay.querySelector('#veta-theme-default');
    const themePink    = overlay.querySelector('#veta-theme-pink');
    const themeIkea    = overlay.querySelector('#veta-theme-ikea');

    const currentTheme = settings.themeAccent || 'default';
    if (themeDefault) themeDefault.checked = currentTheme === 'default';
    if (themePink)    themePink.checked    = currentTheme === 'pink';
    if (themeIkea)    themeIkea.checked    = currentTheme === 'ikea';

    if (showExtraMenu)    showExtraMenu.checked    = settings.showExtraMenu   !== false;
    if (showExtraRatio)   showExtraRatio.checked   = settings.showExtraRatio  !== false;
    if (showExtraQuality) showExtraQuality.checked = settings.showExtraQuality!== false;
    if (showExtraJson)    showExtraJson.checked    = settings.showExtraJson   !== false;

    if (showBotAnswerBox) showBotAnswerBox.checked = settings.showBotAnswerBox !== false;
    if (showEditorTools)  showEditorTools.checked  = settings.showEditorToolsTab !== false;
    if (showFavourites)   showFavourites.checked   = settings.showFavourites !== false;

        if (showBotAnswerBox) showBotAnswerBox.checked = settings.showBotAnswerBox !== false;
    if (showEditorTools)  showEditorTools.checked  = settings.showEditorToolsTab !== false;
    if (showFavourites)   showFavourites.checked   = settings.showFavourites !== false;

    if (showChatbotPreview)  showChatbotPreview.checked  = settings.showChatbotPreview !== false;
    if (showChatbotArticles) showChatbotArticles.checked = settings.showChatbotArticles !== false;
    if (chatbotIconInput)    chatbotIconInput.value      = settings.chatbotProfileIcon || '';

      

    function reapplyDocumentTools() {
      try { ensureNoEnRatioButton(); } catch {}
    }

    function reapplyEditorTools() {
      try { addEditorToolsTab(); } catch {}
    }

    function reapplyFavourites() {
      try { addFavouriteButton(); } catch {}
      try { updateHeaderUtilities(); } catch {}
    }
if (chatbotIconInput) {
  chatbotIconInput.value = settings.chatbotProfileIcon || '';
  chatbotIconInput.style.pointerEvents = 'auto';
  chatbotIconInput.style.position = 'relative';
  chatbotIconInput.style.zIndex = '2';

  const keepFocus = (e) => {
    e.stopPropagation();
  };

  chatbotIconInput.addEventListener('pointerdown', keepFocus);
  chatbotIconInput.addEventListener('click', keepFocus);

  const updateIcon = () => {
    const val = (chatbotIconInput.value || '').trim();
    saveVetaSettings({ chatbotProfileIcon: val }, () => {
      applyAvatarEmojiFromSettings();
    });
  };

  chatbotIconInput.addEventListener('blur', updateIcon);

  chatbotIconInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      updateIcon();
    }
  });

  // Auto-focus when settings open to make it easier to enter an emoji
  setTimeout(() => {
    try {
      chatbotIconInput.focus();
    } catch (e) {}
  }, 50);
}


    // Theme change handler
    function setTheme(value) {
      saveVetaSettings({ themeAccent: value }, () => {
        applyVetaThemeFromSettings();
      });
    }

    [themeDefault, themePink, themeIkea].forEach(radio => {
      if (!radio) return;
      radio.addEventListener('change', () => {
        if (radio.checked) {
          setTheme(radio.value);
        }
      });
    });

    if (showExtraMenu) {
      showExtraMenu.addEventListener('change', () => {
        saveVetaSettings({ showExtraMenu: !!showExtraMenu.checked }, reapplyDocumentTools);
      });
    }
    if (showExtraRatio) {
      showExtraRatio.addEventListener('change', () => {
        saveVetaSettings({ showExtraRatio: !!showExtraRatio.checked }, reapplyDocumentTools);
      });
    }
    if (showExtraQuality) {
      showExtraQuality.addEventListener('change', () => {
        saveVetaSettings({ showExtraQuality: !!showExtraQuality.checked }, reapplyDocumentTools);
      });
    }
    if (showExtraJson) {
      showExtraJson.addEventListener('change', () => {
        saveVetaSettings({ showExtraJson: !!showExtraJson.checked }, reapplyDocumentTools);
      });
    }

    if (showBotAnswerBox) {
      showBotAnswerBox.addEventListener('change', () => {
        saveVetaSettings({ showBotAnswerBox: !!showBotAnswerBox.checked }, () => {
          applyBotAnswerVisibilityFromSettings();
        });
      });
    }

    if (showEditorTools) {
      showEditorTools.addEventListener('change', () => {
        saveVetaSettings({ showEditorToolsTab: !!showEditorTools.checked }, () => {
          reapplyEditorTools();
        });
      });
    }

    if (showFavourites) {
      showFavourites.addEventListener('change', () => {
        saveVetaSettings({ showFavourites: !!showFavourites.checked }, () => {
          reapplyFavourites();
        });
      });
    }

        if (showChatbotPreview) {
      showChatbotPreview.addEventListener('change', () => {
        saveVetaSettings({ showChatbotPreview: !!showChatbotPreview.checked }, () => {
          applyChatbotPreviewVisibilityFromSettings();
        });
      });
    }

    if (showChatbotArticles) {
      showChatbotArticles.addEventListener('change', () => {
        saveVetaSettings({ showChatbotArticles: !!showChatbotArticles.checked }, () => {
          ensureChatbotArticlesMenu();
        });
      });
    }

 

    // Close handling
    const closeButtons = overlay.querySelectorAll('.veta-settings-close');
    function closeOverlay() {
      overlay.style.display = 'none';
    }
    closeButtons.forEach(btn => {
      btn.addEventListener('click', closeOverlay);
    });
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) {
        closeOverlay();
      }
    }, { once: true });

    // Learn more link – set real URL later
    const learnMoreLink = overlay.querySelector('#veta-settings-learn-more');
    if (learnMoreLink) {
      // TODO: when you have your SharePoint/OneDrive doc, paste URL here:
      learnMoreLink.href = '#';
    }
  });
}





// Because the drawer is dynamically rendered, use a MutationObserver
const extensionSettingsObserver = new MutationObserver(() => {
  try { injectExtensionSettingsItem(); } catch {}
});

// Start observing the full document (like you do elsewhere)
(function safeObserveExtensionSettings() {
  try {
    const root = document.documentElement || document.body || null;
    if (root && root.nodeType === 1) {
      extensionSettingsObserver.observe(root, { childList: true, subtree: true });
      return;
    }
    setTimeout(safeObserveExtensionSettings, 50);
  } catch {
    setTimeout(safeObserveExtensionSettings, 100);
  }
})();

function applyBotAnswerVisibilityFromSettings() {
  // Hvis settings ikke er lastet enda, hent dem først
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { applyBotAnswerVisibilityFromSettings(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const host = document.getElementById('veta-botanswer-host');
  if (!host) return; // boksen er ikke laget enda

  if (settings.showBotAnswerBox === false) {
    host.style.display = 'none';
  } else {
    host.style.display = '';
  }
}

function applyChatbotPreviewVisibilityFromSettings() {
  // Last settings hvis ikke er gjort enda
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { applyChatbotPreviewVisibilityFromSettings(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const host = document.getElementById('veta-botanswer-host');
  if (!host || !host.shadowRoot) return;

  const btn = host.shadowRoot.querySelector('.btn.btn-chat');
  if (!btn) return;

  if (settings.showChatbotPreview === false) {
    btn.style.display = 'none';
  } else {
    btn.style.display = '';
  }
}

let _vetaLastAppliedTheme = null;


function applyVetaTheme(theme) {
  // If it's the same theme as last time, do nothing
  if (theme === _vetaLastAppliedTheme) {
    return;
  }
  if (!['default', 'pink', 'ikea'].includes(theme)) {
    theme = 'default';
  }
  _vetaLastAppliedTheme = theme;

  let styleEl = document.getElementById('veta-theme-style');

  if (!styleEl) {
    const head = document.head || document.getElementsByTagName('head')[0] || document.documentElement;
    if (!head) {
      // DOM not ready yet
      return;
    }
    styleEl = document.createElement('style');
    styleEl.id = 'veta-theme-style';
    head.appendChild(styleEl);
  }

  let css = '';

  // DEFAULT – ingen overrides, Veta kjører “vanlig”
  if (theme === 'default') {
    css = '';
  }

  // 💗 COSY PINK
  else if (theme === 'pink') {
    css = `
      body {
        background-color: #fff7fb !important;
      }

      .cwds-header__header {
        background: linear-gradient(90deg, #ff80b5, #ff9ac7) !important;
        border-bottom: 3px solid #ffcee4 !important;
        color: #2b0b19 !important;
      }

      main,
      .cwds-layout__main,
      .skill-wrap,
      .tabs__panel,
        background-color: #fff7fb !important;
      }

      .list-view-item__action {
        background-color: #ffffff !important;
        border: 1px solid #ffd6e8 !important;
      }

      .veta-extra-btn,
      .veta-extra-menu .btn {
        background-color: #ff80b5 !important;
        border-color: #e7548f !important;
        color: #2b0b19 !important;
      }
    `;
  }

  // 💙 IKEA (som før, bare samlet her)
  else if (theme === 'ikea') {
    css = `
      .cwds-header__header {
        background-color: #0058a3 !important; /* IKEA blå */
        border-bottom: 3px solid #ffda1a !important; /* IKEA gul */
      }

      body {
        background-color: #f5f7fa !important;
      }

      #veta-botanswer-host .box {
        border-color: #0058a3 !important;
      }

      .veta-extra-btn,
      .veta-extra-menu .btn {
        background-color: #0058a3 !important;
        border-color: #00427a !important;
        color: #ffffff !important;
      }
    `;
  }

  styleEl.textContent = css;
}

function applyVetaThemeFromSettings() {
  // Make sure settings are loaded first
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { applyVetaThemeFromSettings(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const accent = settings.themeAccent || 'default';

  applyVetaTheme(accent);
}

// Initial load: fetch settings and apply theme ASAP
loadVetaSettings(() => {
  try {
    applyVetaThemeFromSettings();
  } catch (e) {
    console.warn('[Veta] Failed to apply theme on init:', e);
  }
});

// Find a stable key to use for notes for the currently open article
function getVetaCurrentDocKey(root) {
  if (!root) root = document;

  // 1) Try GUID from "open article in new window" link
  const openLink = root.querySelector('a[href*="/document/"]');
  if (openLink) {
    const href = openLink.getAttribute('href') || '';
    const m = href.match(/\/document\/([^?]+)/);
    if (m && m[1]) {
      return 'guid:' + m[1];
    }
  }

  // 2) Fallback: numeric id near the header (the grey number, e.g. 14948)
  const idSpan = root.querySelector('.document-card-header .text-sm.text-gray-400');
  if (idSpan) {
    const raw = (idSpan.textContent || '').trim();
    if (raw) return 'id:' + raw;
  }

  return null;
}


function filterChatbotDocs(docs, term, useRegex) {
  if (!term) return docs;

  let tester;
  if (useRegex) {
    let re;
    try {
      re = new RegExp(term, 'i');
    } catch (e) {
      // Ugyldig regex => ingen treff
      return [];
    }
    tester = (v) => v && re.test(String(v));
  } else {
    const lower = term.toLowerCase();
    tester = (v) => v && String(v).toLowerCase().includes(lower);
  }

  return docs.filter(doc => {
    const kws = Array.isArray(doc.keywords) ? doc.keywords : [];
    return (
      tester(doc.title) ||
      tester(doc.id) ||
      tester(doc.guid) ||
      kws.some(tester)
    );
  });
}



function ensureChatbotArticlesMenu() {
  if (!location.pathname.startsWith('/editor')) return;

  // Sørg for at settings er lastet først
  if (!window.__vetaSettingsLoaded && typeof loadVetaSettings === 'function') {
    loadVetaSettings(() => {
      try { ensureChatbotArticlesMenu(); } catch {}
    });
    return;
  }

  const settings = window.__vetaSettings || VETA_DEFAULT_SETTINGS;
  const enabled = settings.showChatbotArticles !== false;

  const aside = document.querySelector('aside');
  if (!aside) return;

  const nav = aside.querySelector('nav');
  if (!nav) return;

  // Fjern ev. gammel/feil-plassert chatbot-link UTENFOR nav
  const existingOutside = aside.querySelector('.veta-chatbot-menu-item');
  if (existingOutside && !nav.contains(existingOutside)) {
    existingOutside.remove();
  }

  const existingInNav = nav.querySelector('.veta-chatbot-menu-item');

  // Hvis funksjonen er skrudd AV → fjern og stopp
  if (!enabled) {
    if (existingInNav) existingInNav.remove();
    return;
  }

  // Hvis den allerede finnes inne i nav, ikke lag en til
  if (existingInNav) return;

  // Bruk et eksisterende menypunkt som mal – helst "Artikkelliste"
  const template =
    nav.querySelector('a[href="/editor/document"]') || // Artikkelliste
    nav.querySelector('a'); // fallback: første menypunkt

  if (!template) return;

  const link = template.cloneNode(true);
  link.classList.add('veta-chatbot-menu-item');

  // Vi vil ikke navigere bort – vi håndterer klikket selv
  link.removeAttribute('href');

  // Sett label-tekst
  const label = link.querySelector('.text');
  if (label) {
    label.textContent = 'Chatbot articles';
  } else {
    // fallback hvis strukturen endrer seg en dag
    link.textContent = 'Chatbot articles';
  }

  link.addEventListener('click', function (e) {
    e.preventDefault();
    e.stopPropagation();
    showChatbotArticlesView();
  });

  // Legg til helt nederst i menyen (under Hurtiglenker)
  nav.appendChild(link);
}





// ===========================
// Chatbot articles (editor)
// ===========================

var __vetaChatbotDocs = null;
var __vetaChatbotDocsMissing = false;
var __vetaChatbotDocsUrl = null;

function loadChatbotDocs() {
  if (__vetaChatbotDocs) {
    return Promise.resolve(__vetaChatbotDocs);
  }

  if (!__vetaChatbotDocsUrl) {
    try {
      __vetaChatbotDocsUrl = chrome && chrome.runtime
        ? chrome.runtime.getURL('Public veta documents list.json')
        : null;
    } catch {
      __vetaChatbotDocsUrl = null;
    }
  }

  function normalizeDocs(data) {
    if (!Array.isArray(data)) {
      console.warn('[Veta chatbot] VETA_CHATBOT_DOCS is not an array');
      return [];
    }
    return data.filter(function (doc) {
      return Array.isArray(doc.keywords) && doc.keywords.length > 0;
    });
  }

  return new Promise(function (resolve) {
    if (!__vetaChatbotDocsUrl) {
      var fallback = window.VETA_CHATBOT_DOCS;
      __vetaChatbotDocsMissing = !Array.isArray(fallback) || fallback.length === 0;
      __vetaChatbotDocs = normalizeDocs(fallback || []);
      resolve(__vetaChatbotDocs);
      return;
    }

    fetch(__vetaChatbotDocsUrl)
      .then(function (res) { return res.json(); })
      .then(function (data) {
        __vetaChatbotDocsMissing = !Array.isArray(data) || data.length === 0;
        __vetaChatbotDocs = normalizeDocs(data || []);
        resolve(__vetaChatbotDocs);
      })
      .catch(function () {
        var fallback = window.VETA_CHATBOT_DOCS;
        __vetaChatbotDocsMissing = !Array.isArray(fallback) || fallback.length === 0;
        __vetaChatbotDocs = normalizeDocs(fallback || []);
        resolve(__vetaChatbotDocs);
      });
  });
}


function openChatbotDocPanel(docUrl, docTitle) {
  if (!docUrl) return;

  let panel = document.getElementById('veta-chatbot-panel');
  let isNew = false;

  if (!panel) {
    isNew = true;
    panel = document.createElement('div');
    panel.id = 'veta-chatbot-panel';
    document.body.appendChild(panel);

    Object.assign(panel.style, {
      position: 'fixed',
      top: '0',
      right: '0',
      height: '100vh',
      width: '65rem',              // juster bredde om du vil
      maxWidth: '100%',
      background: '#ffffff',
      boxShadow: '0 0 30px rgba(0,0,0,0.25)',
      zIndex: '9999',
      display: 'flex',
      flexDirection: 'column',
      transform: 'translateX(100%)',
      transition: 'transform 0.25s ease-out'
    });

    panel.innerHTML = `
      <div style="
        padding: 10px 10px;
        border-bottom: 1px solid #e5e5e5;
        display:flex;
        align-items:center;
        justify-content:space-between;
        gap: 8px;
        font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif;
      ">
        <div id="veta-chatbot-panel-title" style="font-size:14px; font-weight:600;">
          Document
        </div>
        <button type="button" id="veta-chatbot-panel-close" style="
          border-radius: 999px;
          border: 1px solid #ccc;
          padding: 2px 8px;
          font-size: 12px;
          background:#f9f9f9;
          cursor:pointer;
        ">
          Close
        </button>
      </div>
      <div style="flex:1; min-height:0;">
        <iframe id="veta-chatbot-panel-iframe"
          src=""
          style="border:0; width:100%; height:100%;"></iframe>
      </div>
    `;

    const closeBtn = panel.querySelector('#veta-chatbot-panel-close');
    closeBtn.addEventListener('click', () => {
      panel.style.transform = 'translateX(100%)';
    });
  }

  const titleEl = panel.querySelector('#veta-chatbot-panel-title');
  const iframeEl = panel.querySelector('#veta-chatbot-panel-iframe');

  if (titleEl) {
    titleEl.textContent = docTitle || 'Document';
  }
  if (iframeEl && iframeEl.src !== docUrl) {
    iframeEl.src = docUrl;
  }

  if (isNew) {
    setTimeout(() => {
      panel.style.transform = 'translateX(0)';
    }, 10);
  } else {
    panel.style.transform = 'translateX(0)';
  }
}


function getMarketParams(docs) {
  if (!Array.isArray(docs) || docs.length === 0) {
    return { lang: 'no', market: 'no' };
  }
  const market = String(docs[0].marketCode || 'NO').toLowerCase();
  const map = {
    no: { lang: 'no', market: 'no' },
    se: { lang: 'sv', market: 'se' },
    dk: { lang: 'da', market: 'dk' },
    fi: { lang: 'fi', market: 'fi' }
  };
  return map[market] || { lang: market, market };
}

function renderList(items, listEl, countEl) {
  if (!items || !items.length) {
    countEl.textContent = 'No results';
    listEl.innerHTML = `<div style="font-size:13px; color:#777;">No articles matched the search.</div>`;
    return;
  }

  countEl.textContent = items.length + ' results';
  listEl.innerHTML = '';

  const marketParams = getMarketParams(items);

  items.forEach(doc => {
  const row = document.createElement('div');
row.style.padding = '8px 16px';
row.style.margin = '0 4px';
row.style.borderBottom = '1px solid #eee';
row.style.borderRadius = '6px';
row.style.cursor = 'pointer';


    const kws = Array.isArray(doc.keywords) ? doc.keywords.join(', ') : '';

    const market = marketParams.market;
    const lang = marketParams.lang;
    const docUrl = doc.guid
      ? `https://km.ingka.com/document/${doc.guid}?lang=${lang}&market=${market}`
      : '';

    row.innerHTML = `
      <div style="display:flex; justify-content:space-between; gap:12px; align-items:flex-start;">
        <div>
          <div style="font-weight:600; margin-bottom:2px;">
            ${doc.title || '(untitled)'}
          </div>
          <div style="font-size:12px; color:#666; margin-bottom:2px;">
            ID: ${doc.id || '-'}
          </div>
          ${kws ? `
            <div style="font-size:12px; color:#444; margin-top:2px;">
              <strong>Keywords:</strong> ${kws}
            </div>
          ` : ''}
        </div>
      </div>
    `;

    if (docUrl) {
      row.dataset.docUrl = docUrl;
      row.dataset.docTitle = doc.title || doc.id || 'Document';
      row.addEventListener('click', () => {
        openChatbotDocPanel(row.dataset.docUrl, row.dataset.docTitle);
      });
    }

    listEl.appendChild(row);
  });
}



function filterChatbotDocs(docs, term, useRegex) {
  if (!term) return docs;

  var tester;
  if (useRegex) {
    var re;
    try {
      re = new RegExp(term, 'i');
    } catch (e) {
      // ugyldig regex => ingen treff
      return [];
    }
    tester = function (v) { return v && re.test(String(v)); };
  } else {
    var lower = term.toLowerCase();
    tester = function (v) { return v && String(v).toLowerCase().includes(lower); };
  }

  return docs.filter(function (doc) {
    var kws = Array.isArray(doc.keywords) ? doc.keywords : [];
    return (
      tester(doc.title) ||
      tester(doc.id) ||
      tester(doc.guid) ||
      kws.some(tester)
    );
  });
}

function getEditorMainSection() {
  // hoveddelen du limte inn
  var section = document.querySelector('section.ml-10.flex-1');
  if (!section) {
    // fallback hvis klassenavnet endres litt
    section = document.querySelector('section.ml-10') || document.querySelector('section.flex-1');
  }
  return section;
}

function showChatbotArticlesView() {
  var mainSection = getEditorMainSection();
  if (!mainSection) {
    console.warn('[Veta chatbot] fant ikke main-section for editor');
    return;
  }

  // Bygg ny layout inspirert av Artikkelliste
  mainSection.innerHTML = `
    <div class="relative grid grid-cols-1 px-10">
      <div class="p-5">
        <div class="mb-4 flex items-center justify-between">
          <h1 class="text text--headline-m">Chatbot articles</h1>
          <p class="text text--body-s text-grey-500">
            Search articles linked to Dialogflow / keywords
          </p>
        </div>

        <div class="mb-4 flex space-x-4">
          <div class="input-field label-wrapper label-wrapper--text-input w-[30%]">
            <label class="input-field__label" for="chatbot-search-input">
              Search in title / id / guid / keywords
            </label>
            <div class="input-field__wrapper">
              <input id="chatbot-search-input" type="text" value="" />
              <span class="validation-border"></span>
            </div>
          </div>

          <label class="flex items-center gap-2 text text--body-s">
            <input id="chatbot-regex-toggle" type="checkbox" />
            <span>Use regex</span>
          </label>
        </div>

        <div class="my-3 flex items-center justify-between">
          <div class="flex items-center gap-3">
            <p id="chatbot-result-count">Loading data...</p>
          </div>
        </div>

        <div class="overflow-hidden rounded-md bg-white shadow">
          <div class="max-h-[600px] overflow-auto">
            <div id="chatbot-results-root">
              <div class="grid grid-cols-[5rem_1fr_10rem_18rem] border-b bg-gray-50 p-4 font-bold">
                
                <div class="flex items-center whitespace-nowrap">
                  <div class="text text--heading-xs">Document list - Public documents that contain keywords</div>
                </div>
              
                
              </div>
              <div id="chatbot-results-list"></div>
            </div>
          </div>
        </div>
      </div>
    </div>
  `;

  var input = document.getElementById('chatbot-search-input');
var regexToggle = document.getElementById('chatbot-regex-toggle');
var countEl = document.getElementById('chatbot-result-count');
var listEl = document.getElementById('chatbot-results-list');

function runSearch(docs) {
  var term = (input.value || '').trim();
  var useRegex = !!regexToggle.checked;
  var filtered = filterChatbotDocs(docs, term, useRegex);
  renderList(filtered, listEl, countEl);
}

loadChatbotDocs().then(function (docs) {
  if (__vetaChatbotDocsMissing && window.showCopyToast) {
    window.showCopyToast('Provide JSON file with document overview.');
  }

  // first view: show all
  renderList(docs, listEl, countEl);

  input.addEventListener('input', function () {
    runSearch(docs);
  });
  regexToggle.addEventListener('change', function () {
    runSearch(docs);
  });
});


}






/* ===========================
   Startup + SPA resilience
   =========================== */

if (typeof window.__docRightHidden !== 'boolean') {
  window.__docRightHidden = false;
}

// Initial delayed runs (your page often needs a small delay)
setTimeout(addFavouriteButton, 750);
setTimeout(updateHeaderUtilities, 750);
setTimeout(ensureDocRightToggleButton, 750);
setTimeout(cleanupEditorBreadcrumbText, 750);
setTimeout(ensureNoEnRatioButton, 750);
setTimeout(removeLiveEditWarning, 750);
setTimeout(ensureEditorHeaderStyle, 750); 
setTimeout(applyVetaThemeFromSettings, 750);
setTimeout(ensureChatbotArticlesMenu, 750); 
setTimeout(applyAvatarEmojiFromSettings, 1500);
setTimeout(applyChatbotPreviewVisibilityFromSettings, 750);





// Debounced SPA observer for header/buttons
let debounceTimer;
const spaObserver = new MutationObserver(() => {
  clearTimeout(debounceTimer);
  debounceTimer = setTimeout(() => {
  try { addFavouriteButton(); } catch {}
  try { updateHeaderUtilities(); } catch {}
  try { ensureDocRightToggleButton(); } catch {}
  try { cleanupEditorBreadcrumbText(); } catch {}
  try { ensureNoEnRatioButton(); } catch {}
  try { addEditorToolsTab(); } catch {}  
  try { removeLiveEditWarning(); } catch {}
  try { ensureEditorHeaderStyle(); } catch {}
  try { applyBotAnswerVisibilityFromSettings(); } catch {}
  try { applyVetaThemeFromSettings(); } catch {}
  try { ensureVetaNotesTab(); } catch {}
  try { ensureChatbotArticlesMenu(); } catch {}
  try { applyAvatarEmojiFromSettings(); } catch {}
  try { applyChatbotPreviewVisibilityFromSettings(); } catch {}

  




  if (typeof addUpdateButton === 'function') {
    try { addUpdateButton(); } catch {}
  }
}, 500);
});

// Safe observe (prevents "parameter 1 is not of type 'Node'")
(function safeObserveHeader() {
  try {
    const root = document.documentElement || document.body || null;
    if (root && root.nodeType === 1) {
      spaObserver.observe(root, { childList: true, subtree: true });
      return;
    }
    setTimeout(safeObserveHeader, 50);
  } catch {
    setTimeout(safeObserveHeader, 100);
  }
})();



